
/**************************************************************************
 *
 *  $Id: timeutil.c 1.1.1.5 2017/04/10 12:37:46Z martin TEST martin.burnicki $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Meinberg Library module for safe time conversion routines.
 *
 * -----------------------------------------------------------------------
 *  $Log: timeutil.c $
 *  Revision 1.1.1.5  2017/04/10 12:37:46Z  martin
 *  Fixed some compiler warnings.
 *  Revision 1.1.1.4  2017/02/06 11:45:57Z  martin
 *  Fixed build under Windows, and implemented mbg_clock_settime(),
 *  but system time is actually not yet set.
 *  Revision 1.1.1.3  2016/11/21 15:22:37Z  martin
 *  Revision 1.1.1.2  2016/08/11 15:09:46Z  martin
 *  *** empty log message ***
 *  Revision 1.1.1.1  2016/08/11 07:54:40  martin
 *  Fixed build with old Visual Studio.
 *  Revision 1.1  2016/07/15 14:14:19Z  martin
 *  Initial revision
 *
 **************************************************************************/

#define _TIMEUTIL
 #include <timeutil.h>
#undef _TIMEUTIL

#include <str_util.h>
#include <mbgerror.h>

#if defined( MBG_TGT_WIN32 )
  #include <stdio.h>
#endif



/*HDR*/
size_t snprint_gmtime_error( char *s, size_t max_len, int mbg_errno, time_t t, const char *calling_fnc )
{
  size_t n = snprintf_safe( s, max_len, "gmtime() call failed" );

  if ( calling_fnc )
    n += snprintf_safe( &s[n], max_len - n, " in %s", calling_fnc );

  #if defined( _MSC_VER ) && ( _MSC_VER < 1500 )
    //### TODO E.g. in VC6 time_t is only 32 bit anyway.
    n += snprintf_safe( &s[n], max_len - n, " for time_t %lu: %s",
                        (unsigned long) t, mbg_strerror( mbg_errno ) );
  #else
    n += snprintf_safe( &s[n], max_len - n, " for time_t %llu: %s",
                        (unsigned long long) t, mbg_strerror( mbg_errno ) );
  #endif

  return n;

}  // snprint_gmtime_error



#if defined( MBG_TGT_WIN32 )

typedef int clockid_t;
#define clockid_t clockid_t

#define CLOCK_REALTIME  ( (clockid_t) 0 )

/*HDR*/
int mbg_clock_gettime( clockid_t clock_id, struct timespec *tp )
{
  if ( clock_id == CLOCK_REALTIME )
  {
    #if defined( TIME_UTC )  // C11 / VS2015+
      int rc = timespec_get( tp, TIME_UTC );  // TODO Check this code
      return ( rc == 0 ) ? -1 : 0   // rc == 0 means error
    #else
      #define EPOCH_HNS     116444736000000000i64
      #define NSEC_PER_SEC          1000000000i64
      FILETIME ft;
      unsigned __int64 tmp;
      gstaft_fnc( &ft );
      tmp = ( (__int64) ft.dwHighDateTime << 32 ) | ft.dwLowDateTime;
      tmp -= EPOCH_HNS;   // convert to Unix epoch
      tmp *= 100;         // convert to nanoseconds
      tp->tv_sec = ( tmp / NSEC_PER_SEC );
      tp->tv_nsec = ( tmp % NSEC_PER_SEC );
      return 0;
    #endif
  }
  else
    return -1;    // TODO this is e.g. in case of CLOCK_MONOTONIC, we could use QPC then.

}  // mbg_clock_gettime



/*HDR*/
int mbg_clock_settime( clockid_t clock_id, const struct timespec *tp )
{
  if ( clock_id == CLOCK_REALTIME )
  {
#if 0  // ### TODO FIXME This needs to be implemented.
    #if defined( TIME_UTC )  // C11 / VS2015+
      int rc = timespec_get( res, TIME_UTC );  // TODO Check this code
      return ( rc == 0 ) ? -1 : 0   // rc == 0 means error
    #else
      #define EPOCH_HNS     116444736000000000i64
      #define NSEC_PER_SEC          1000000000i64
      FILETIME ft;
      unsigned __int64 tmp;
      gstaft_fnc( &ft );
      tmp = ( (__int64) ft.dwHighDateTime << 32 ) | ft.dwLowDateTime;
      tmp -= EPOCH_HNS;   // convert to Unix epoch
      tmp *= 100;         // convert to nanoseconds
      res->tv_sec = ( tmp / NSEC_PER_SEC );
      res->tv_nsec = ( tmp % NSEC_PER_SEC );
      return 0;
    #endif
#endif

    return 0;  // FIXME this is actually not true
  }
  else
    return -1;    // TODO this is e.g. in case of CLOCK_MONOTONIC, we could use QPC then.

}  // mbg_clock_settime



bool force_legacy_gstaft; 


/*HDR*/
void check_precise_time_api( void )
{
  const char *info ="";
  GSTAFT_FNC tmp_fnc;
  HINSTANCE h = LoadLibrary( "kernel32.dll" );

  if ( h == NULL )
  {
    info = "Precise system time may not be supported; failed to get handle for kernel32.dll.";
    goto out;
  }

  tmp_fnc = (GSTAFT_FNC) GetProcAddress( h, "GetSystemTimePreciseAsFileTime" );

  if ( tmp_fnc == NULL )
  {
    info = "Precise system time NOT supported";
    goto out;
  }

  if ( force_legacy_gstaft )
  {
    info = "Precise system time is supported, but legacy function used by request";
    goto out;
  }

  gstaft_fnc = tmp_fnc;
  info = "Precise system time is supported and used";

out:
  printf( "%s\n", info );

}  // check_precise_time_api

#endif

