
/**************************************************************************
 *
 *  $Id: gpsserio.c 1.13.1.16 2016/11/15 15:43:44Z martin TEST $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Low level functions used to access Meinberg GPS receivers via
 *    serial port.
 *
 * -----------------------------------------------------------------------
 *  $Log: gpsserio.c $
 *  Revision 1.13.1.16  2016/11/15 15:43:44Z  martin
 *  Account for modified mbgserio functions.
 *  Revision 1.13.1.15  2016/06/29 11:59:03  philipp
 *  Extended socket API by TCP client
 *  Revision 1.13.1.14  2016/06/03 11:10:28  thomas-b
 *  Fixed Windows compatibility issues in xmt_tbuff
 *  Revision 1.13.1.13  2016/03/23 13:45:47  thomas-b
 *  Fixed check for buffer overflow and return appropriate rc
 *  Revision 1.13.1.12  2016/03/21 13:27:32  martin
 *  *** empty log message ***
 *  Revision 1.13.1.11  2016/03/18 06:15:30  thomas-b
 *  Fixed call of mbg_memcpy_reversed
 *  Revision 1.13.1.10  2016/03/17 15:54:59  martin
 *  Use mbg_memcpy..() functions.
 *  Revision 1.13.1.9  2016/03/11 10:16:38  thomas-b
 *  Added function chk_hdr_rcvd
 *  Revision 1.13.1.8  2015/12/10 16:30:15  martin
 *  *** empty log message ***
 *  Revision 1.13.1.7  2015/12/09 10:21:22  martin
 *  *** empty log message ***
 *  Revision 1.13.1.6  2015/11/02 10:12:26  martin
 *  *** empty log message ***
 *  Revision 1.13.1.5  2015/09/15 15:31:52  martin
 *  Include sys/time.h to fix a compiler warning.
 *  Revision 1.13.1.4  2015/09/08 14:51:59  martin
 *  Added some comments regarding mutex support.
 *  Revision 1.13.1.3  2015/07/24 11:07:17  martin
 *  Revision 1.13.1.2  2015/05/20 12:21:23  martin
 *  Revision 1.13.1.1  2015/05/19 13:20:21  daniel
 *  Preliminary support for USB_DIRECT_IO
 *  Revision 1.13  2015/05/13 13:47:52  martin
 *  Support new libusb v1.0 and newer which is not compatible with v0.1x.
 *  Use preprocessor symbol MBG_TGT_POSIX.
 *  Use Meinberg error codes from mbgerror.h.
 *  Support XBP addressing.
 *  Updated doxygen-style comments.
 *  Revision 1.12  2012/03/08 12:09:38  martin
 *  Removed an obsolete printf().
 *  Revision 1.11  2012/03/06 15:41:24Z  martin
 *  Use mbgserio_read/write functions rather than macros.
 *  Save system timestamp on start of incoming binary packet.
 *  Revision 1.10  2011/07/29 09:53:43Z  daniel
 *  Account for communication via USB if _USE_USB_IO is defined
 *  Revision 1.9  2009/09/01 09:51:56  martin
 *  Removed obsolete includes.
 *  Revision 1.8  2009/03/10 16:58:09  martin
 *  Fixed compiler warnings.
 *  Revision 1.7  2008/09/03 15:22:40  martin
 *  Decryption with wrong password yields garbage, still needs to be fixed.
 *  In xmt_tbuff() use MBG_PORT_HANDLE for serial connections.
 *  Some cleanup in check_transfer().
 *  Fixed a VC6 compiler warning.
 *  Moved low level serial I/O routines to mbgserio.c.
 *  Revision 1.6  2006/10/25 12:24:01Z  martin
 *  Support serial I/O under Windows.
 *  Removed obsolete code.
 *  Revision 1.5  2006/08/24 12:57:41Z  martin
 *  Conditionally also support network socket I/O and encrypted packets.
 *  Support also serial I/O conditionally only.
 *  Added/renamed/redefined structures as required.
 *  Revision 1.4  2006/05/17 10:19:39  martin
 *  Account for renamed structure.
 *  Revision 1.3  2005/04/26 11:00:30  martin
 *  Added standard file header.
 *  Source code cleanup.
 *  check_transfer() now expects a control structure which keeps
 *  the reception status corresponding to the receive buffer.
 *  After receive error reinitialize the byte counter to restart
 *  reception with next incoming byte.
 *  Use type CSUM where appropriate.
 *  Use C99 fixed-size data types where applicable.
 *  Renamed the function csum() to mbg_csum().
 *
 **************************************************************************/

#define _GPSSERIO
 #include <gpsserio.h>
#undef _GPSSERIO

#include <mbgerror.h>
#include <str_util.h>

#include <string.h>
#include <stdlib.h>
#include <stddef.h>

#if defined( MBG_TGT_POSIX )
  #include <unistd.h>
  #include <sys/time.h>
#endif

#if _USE_ENCRYPTION
  #include <aes128.h>
#endif



#if _USE_SERIAL_IO_FTDI

/*HDR*/
/**
 * @brief Translate a FTDI API status code to one of the @ref MBG_ERROR_CODES
 *
 * @param[in] status  A status code defined by the FTDI API interface
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
int mbg_ftdi_ft_status_to_mbg( FT_STATUS status )
{
  switch ( status )
  {
    case FT_OK:                               return MBG_SUCCESS;
    case FT_INVALID_HANDLE:                   return MBG_ERR_INV_HANDLE;
    case FT_DEVICE_NOT_FOUND:                 return MBG_ERR_NO_DEV;
    case FT_DEVICE_NOT_OPENED:                return MBG_ERR_INV_HANDLE;
    case FT_IO_ERROR:                         return MBG_ERR_IO;
    // case FT_INSUFFICIENT_RESOURCES:           return ;
    case FT_INVALID_PARAMETER:                return MBG_ERR_INV_PARM;
    case FT_INVALID_BAUD_RATE:                return MBG_ERR_INV_PARM;

    // case FT_DEVICE_NOT_OPENED_FOR_ERASE:      return ;
    // case FT_DEVICE_NOT_OPENED_FOR_WRITE:      return ;
    // case FT_FAILED_TO_WRITE_DEVICE:           return ;
    case FT_EEPROM_READ_FAILED:               return MBG_ERR_IO;
    case FT_EEPROM_WRITE_FAILED:              return MBG_ERR_IO;
    // case FT_EEPROM_ERASE_FAILED:              return ;
    // case FT_EEPROM_NOT_PRESENT:               return ;
    // case FT_EEPROM_NOT_PROGRAMMED:            return ;
    case FT_INVALID_ARGS:                     return MBG_ERR_INV_PARM;
    case FT_NOT_SUPPORTED:                    return MBG_ERR_NOT_SUPP_BY_DEV;
    case FT_OTHER_ERROR:                      return MBG_ERR_UNSPEC;
    case FT_DEVICE_LIST_NOT_READY:            return MBG_ERR_NO_DEV;
  }

  return MBG_ERR_UNSPEC;

}  // mbg_ftdi_ft_status_to_mbg

#endif  // _USE_SERIAL_IO_FTDI



/*HDR*/
/**
 * @brief Compute a simple binary checksum
 *
 * Compute a checksum about a number of bytes starting
 * with a given initial value.
 *
 * @param[in] csum  the initial value
 * @param[in] p     pointer to a buffer of data bytes
 * @param[in] n     the number of bytes in the buffer
 *
 * @return the computed checksum
 */
CSUM msg_csum_update( CSUM csum, const uint8_t *p, int n )
{
  int i;

  for ( i = 0; i < n; i++ )
    csum += *p++;

  return csum;

}  /* msg_csum_update */



/*HDR*/
/**
 * @brief Compute a checksum for a binary message
 *
 * @note This function differs from the checksum() function
 * used to compute the checksum of battery-buffered variables.
 *
 * @param[in] p  pointer to a buffer of data bytes
 * @param[in] n  the number of bytes in the buffer
 *
 * @return the computed checksum
 */
CSUM msg_csum( const uint8_t *p, int n )
{
  return msg_csum_update( 0, p, n );

}  /* msg_csum */



/*HDR*/
/**
 * @brief Compute the checksum of a binary message header
 *
 * @param[in] pmh  pointer to a binary message header
 *
 * @return the computed checksum
 */
CSUM msg_hdr_csum( const MSG_HDR *pmh )
{
  return msg_csum( (uint8_t *) pmh,
         sizeof( *pmh ) - sizeof( pmh->hdr_csum ) );

}  /* msg_hdr_csum */



/*HDR*/
/**
 * @brief Check if the header checksum of a binary message is valid
 *
 * @param[in] pmh  pointer to a binary message header
 *
 * @return ::MBG_SUCCESS or ::MBG_ERR_HDR_CSUM
 */
int chk_hdr_csum( const MSG_HDR *pmh )
{
  CSUM calc_csum = msg_hdr_csum( pmh );

  if ( calc_csum != pmh->hdr_csum )
    return MBG_ERR_HDR_CSUM;   /* error */

  return MBG_SUCCESS;

}  /* chk_hdr_csum */



/*HDR*/
/**
 * @brief Check if the data checksum of a binary message is valid
 *
 * @param[in] pmb  pointer to a binary message buffer
 *
 * @return ::MBG_SUCCESS or ::MBG_ERR_DATA_CSUM
 */
int chk_data_csum( const MBG_MSG_BUFF *pmb )
{
  CSUM calc_csum = msg_csum( pmb->u.bytes, pmb->hdr.len );

  if ( calc_csum != pmb->hdr.data_csum )
    return MBG_ERR_DATA_CSUM;   /* error */

  return MBG_SUCCESS;

}  /* chk_data_csum */



/*HDR*/
/**
 * @brief Checks if a binary message header has been completely received
 *
 * @param[in]   prctl   ::MBG_MSG_RCV_CTL that shall be checked
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
int chk_hdr_rcvd( const MBG_MSG_RCV_CTL *prctl )
{
  if ( prctl->rcv_state.cnt == 0 )
    return MBG_ERR_GENERIC;  //### TODO different return code?

  if ( ( prctl->rcv_state.flags & MBG_MSG_RCV_CTL_RCVD_HDR ) == 0 )
    return MBG_ERR_BUSY;

  return MBG_SUCCESS;

} // chk_hdr_rcvd



#if _USE_ENCRYPTION

#ifdef MBG_TGT_WIN32

  static
  void randomize( void )
  {
    LARGE_INTEGER perfc;

    QueryPerformanceCounter( &perfc );

    srand( perfc.LowPart );

  }  /* randomize */

#else

  static
  void randomize( void )
  {
    struct timeval tv;

    gettimeofday( &tv, NULL );

    srand( tv.tv_usec );

  }  /* randomize */

#endif



/*HDR*/
/**
 * @brief Encrypt a binary message
 *
 * In encryption mode the original packet is encrypted and put
 * as data portion into an envelope package.
 *
 * @param[in,out] pmctl Pointer to a valid message control structure
 * @param[in,out] pcmp  Pointer to encryption settings
 * @param[in,out] pmb   Pointer to a binary message buffer
 *
 * @return the number of bytes of the encrypted message, or one of the @ref MBG_ERROR_CODES
 */
int encrypt_message( MBG_MSG_CTL *pmctl, CRYPT_MSG_PREFIX *pcmp, MBG_MSG_BUFF *pmb )
{
  int n_bytes = pmb->hdr.len + sizeof( pmb->hdr );  // size of unencrypted msg
  int rc;

  // align original msg size with 16 byte boundary
  n_bytes += AES_BLOCK_SIZE - ( n_bytes % AES_BLOCK_SIZE );

  // encrypt original message
  rc = aes_encrypt_buff( (uint8_t *) pmb, pmctl->aes_keyvect, pmctl->aes_initvect, n_bytes );

  if ( rc < 0 )  // -1 on error
    return MBG_ERR_ENCRYPT;   // encryption failed


  // copy AES init vector into encrypted message
  //##++ check types of aes_initvect fields
  mbg_memcpy( pcmp->aes_initvect, pmctl->aes_initvect, sizeof( pcmp->aes_initvect ) );

  pcmp->hdr.cmd = GPS_CRYPTED_PACKET;
  pcmp->hdr.len = (uint16_t) ( n_bytes + sizeof( pcmp->aes_initvect ) );

  pcmp->hdr.data_csum = msg_csum( pcmp->aes_initvect, sizeof( pcmp->aes_initvect ) );
  pcmp->hdr.data_csum = msg_csum_update( pcmp->hdr.data_csum,
                                         (uint8_t *) pmb, n_bytes );

  pcmp->hdr.hdr_csum = msg_hdr_csum( &pcmp->hdr );

  return n_bytes;

}  /* encrypt_message */



/*HDR*/
/**
 * @brief Decrypt an encrypted binary message
 *
 * @param[in,out] pmctl Pointer to a valid message control structure
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
int decrypt_message( MBG_MSG_CTL *pmctl )
{
  MBG_MSG_RCV_CTL *prctl = &pmctl->rcv;
  MBG_MSG_BUFF *pmb = prctl->pmb;
  CRYPT_MSG_DATA *pcmd = &pmb->u.crypt_msg_data;
  int rc;
  size_t len;

  if ( pmb->hdr.len < AES_BLOCK_SIZE )
    return 0;

  rc = aes_decrypt_buff( (uint8_t *) &pcmd->std_msg,
                         pmctl->aes_keyvect,
                         pcmd->aes_initvect,
                         pmb->hdr.len - sizeof( pcmd->aes_initvect )
                       );

  if ( rc < 0 )  // decryption error
  {
    prctl->rcv_state.flags |= MBG_MSG_RCV_CTL_DECRYPT_ERR;
    return MBG_ERR_DECRYPT;
  }

  // packet decrypted successfully.
  prctl->rcv_state.flags |= MBG_MSG_RCV_CTL_DECRYPTED;

  // If the wrong password has been used for decryption
  // then decryption may have been formally successful,
  // but the decrypted message contains garbage.
  // So we must check whether the decrypted packet
  // also contains a valid header and data part.
  //##+++++++++  TODO
  // A potential problem arises if the garbage data len exceeds
  // the buffer size and the data csum is computed using
  // that length, which goes beyond the buffer limit.
  // also, me must not copy more than the sizeof MBG_STD_MSG
  // bytes to avoid a buffer overflow.
  // We should really check the decrypted header here.

  len = pcmd->std_msg.hdr.len + sizeof( pcmd->std_msg.hdr );

  if ( len > sizeof( MBG_STD_MSG ) )
    len = sizeof( MBG_STD_MSG );

  // copy the decrypted message to the head of the buffer
  mbg_memcpy( pmb, &pcmd->std_msg, len );


  // now check the CSUMs of the decrypted packet
  //##++++ TODO: which error code to return? This was actually the data portion of the encrypted msg

  if ( chk_hdr_csum( &pmb->hdr ) != MBG_SUCCESS )
    return MBG_ERR_DECRYPT;      // invalid header checksum due to faulty decryption

  if ( chk_data_csum( pmb ) != MBG_SUCCESS )
    return MBG_ERR_DECRYPT;      // invalid data checksum due to faulty decryption

  return MBG_SUCCESS;

}  /* decrypt_message */



/*HDR*/
/**
 * @brief Set communication channel to specified encryption mode
 *
 * @param[in,out] pmctl Pointer to a valid message control structure
 * @param[in]     mode  Encryption mode, usually ::MBG_XFER_MODE_ENCRYPTED
 * @param[in]     key   The crypto key (password) to be used
 */
void set_encryption_mode( MBG_MSG_CTL *pmctl, int mode, const char *key )
{
  int i;

  randomize();

  for ( i = 0; i < AES_BLOCK_SIZE; i++ )
  {
    pmctl->aes_initvect[i] = rand();
    pmctl->aes_keyvect[i] = key[i];
  }

  pmctl->xmt.xfer_mode = mode;

}  // set_encryption_mode

#endif  // _USE_ENCRYPTION



/*HDR*/
/**
 * @brief Complete message header and transmit message
 *
 * Compute checksums and complete the message header, then
 * transmit both header and data. The caller must have copied
 * the data to be sent to the data field of the transmit buffer
 * and have set up the cmd and len fields of the message header.
 *
 * This function doesn't protect access to the device by a mutex,
 * so the calling function has to take care of the mutex handling.
 *
 * @param[in,out] pmctl   Pointer to a valid message control structure
 * @param[in]     p_addr  Pointer to XBP address specifier
 *
 * @return one of the @ref MBG_RETURN_CODES
 */
int xmt_tbuff( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr )
{
  static const char soh = START_OF_HEADER;

  MBG_MSG_BUFF *pmb = pmctl->xmt.pmb;
  int msg_len = sizeof( pmb->hdr ) + pmb->hdr.len;
  int rc = MBG_ERR_UNSPEC;
  #if _USE_ENCRYPTION
    CRYPT_MSG_PREFIX cm_pfx;
  #endif

  // Set up the checksums of the unencrypted packet.
  pmb->hdr.data_csum = msg_csum( pmb->u.bytes, pmb->hdr.len );
  pmb->hdr.hdr_csum = msg_hdr_csum( &pmb->hdr );

  // If an XBP address has been specified we convert
  // the original msg into an XBP msg.
  if ( p_addr )
  {
    // Move the original msg to the data part of an XBP msg which
    // overlaps the source memory area.
    mbg_memcpy_reversed( &pmb->u.xbp_msg_data.std_msg.hdr, &pmb->hdr, msg_len );

    // Fill in the specified XBP address.
    pmb->u.xbp_msg_data.xbp_addr = *p_addr;

    // Set up the new msg header as XPB msg.
    pmb->hdr.cmd = GPS_XBP_PACKET;
    msg_len += sizeof( pmb->u.xbp_msg_data.xbp_addr );
    pmb->hdr.len = msg_len;
    pmb->hdr.data_csum = msg_csum( pmb->u.bytes, pmb->hdr.len );
    pmb->hdr.hdr_csum = msg_hdr_csum( &pmb->hdr );
    msg_len += sizeof( pmb->hdr );
  }

  #if _USE_ENCRYPTION
    if ( pmctl->xmt.xfer_mode == MBG_XFER_MODE_ENCRYPTED )
    {
      memset( &cm_pfx, 0, sizeof( cm_pfx ) );

      rc = encrypt_message( pmctl, &cm_pfx, pmb );
      // on error, one of the MBG_ERROR_CODES, else number of bytes encrypted

      if ( rc < 0 )  // error
        goto out;

      msg_len = rc;
    }
  #endif

  // msg_len now contains the original msg data len which may
  // possibly have been rounded up by the encryption routine.
  //
  // The full msg consists of the CRYPT_MSG_PREFIX (if the msg
  // has been encrypted), the msg header, and msg_len of data.
  switch ( pmctl->conn_type )
  {
    #if _USE_SERIAL_IO
      case MBG_CONN_TYPE_SERIAL:
      {
        MBGSERIO_DEV *mdev = pmctl->st.p_serio;

        // Note: encrypted msgs over serial are not yet supported.

        rc = mbgserio_write( mdev, &soh, sizeof( soh ) );

        if ( rc < 0 )  // error
          goto out;

        if ( rc != sizeof( soh ) )
          goto out_write_failed;

        rc = mbgserio_write( mdev, pmb, msg_len );

        if ( rc < 0 )  // error
          goto out;

        if ( rc != msg_len )  // error
          goto out_write_failed;

      } break;
    #endif  // _USE_SERIAL_IO

    #if _USE_SERIAL_IO_FTDI
      case MBG_CONN_TYPE_SERIAL_FTDI:
      {
        // Yet this is anyway supported under Windows only, so we can
        // safely use Windows-specific types as expected by the FTDI API.
        DWORD bytes_written;
        FT_HANDLE port_handle = pmctl->st.ftdi.port_handle;
        FT_STATUS status;

        // Note: encrypted msgs over serial are not yet supported.
        status = FT_Write( port_handle, (LPVOID) &soh, sizeof( soh ), &bytes_written );

        if ( status != FT_OK )
        {
          rc = mbg_ftdi_ft_status_to_mbg( status );
          goto out;
        }

        if ( bytes_written != sizeof( soh ) )
          goto out_write_failed;

        status = FT_Write( port_handle, pmb, msg_len, &bytes_written );

        if ( status != FT_OK )
        {
          rc = mbg_ftdi_ft_status_to_mbg( status );
          goto out;
        }

        if ( bytes_written != (DWORD) msg_len )
          goto out_write_failed;

      } break;
    #endif  // _USE_SERIAL_IO_FTDI

    #if _USE_SOCKET_IO
      case MBG_CONN_TYPE_SOCKET:
      {
        uint8_t sock_xmt_buffer[sizeof( MBG_MSG_BUFF ) + 1];
        uint8_t *p = sock_xmt_buffer;

        #if defined( DEBUG )
          memset( sock_xmt_buffer, 0, sizeof( sock_xmt_buffer ) );  // only to simplify debugging
        #endif

        *p++ = soh;

        rc = msg_len;  // save the value of msg_len

        #if _USE_ENCRYPTION
          if ( pmctl->xmt.xfer_mode == MBG_XFER_MODE_ENCRYPTED )
          {
            mbg_memcpy( p, &cm_pfx, sizeof( cm_pfx ) );
            p += sizeof( cm_pfx );
            msg_len += sizeof( cm_pfx );
          }
        #endif

        mbg_memcpy( p, pmb, rc );
        p += rc;

        msg_len += sizeof( soh );   // also account for SOH

        rc = sendto( pmctl->st.sockio.sockfd, (const char*)sock_xmt_buffer, msg_len, 0,
                    (const struct sockaddr *) pmctl->st.sockio.p_addr,
                    pmctl->st.sockio.addrlen );

        if ( rc < 0 )  // on error: -1 on Linux, SOCKET_ERROR == -1 on Windows
        {
          rc = mbg_get_last_socket_error( "sendto failed in xmt_tbuff" );
          goto out;
        }

        if ( rc != msg_len )
          goto out_write_failed;

      } break;
  #endif  // _USE_SOCKET_IO

  #if _USE_USB_IO
    case MBG_CONN_TYPE_USB:
    {
      uint8_t usb_xmt_buffer[sizeof( MBG_MSG_BUFF ) + 1];
      uint8_t *p = usb_xmt_buffer;

      #if defined( DEBUG )
        memset( usb_xmt_buffer, 0, sizeof( usb_xmt_buffer ) );  // only to simplify debugging
      #endif

      *p = soh;
      mbg_memcpy( p + 1, pmb, msg_len );
      msg_len++;   // also account for SOH

      // Note: encrypted msgs over USB are not yet supported.
      rc = mbgusbio_write( &pmctl->st.usbio, p, msg_len );

      if ( rc < 0 )  // error
        goto out;

      if ( rc != msg_len )
        goto out_write_failed;

    } break;
  #endif  // _USE_USB_IO

  #if _USE_USB_DIRECT_IO
    case MBG_CONN_TYPE_USB_DIRECT_IO:
    {
      ssize_t bytes;
      uint8_t usb_xmt_buffer[sizeof( MBG_MSG_BUFF ) + 1];
      uint8_t *p = usb_xmt_buffer;

      #if defined( DEBUG )
        memset( usb_xmt_buffer, 0, sizeof( usb_xmt_buffer ) );  // only to simplify debugging
      #endif

      *p = soh;
      mbg_memcpy( p + 1, pmb, msg_len );
      msg_len++;   // also account for SOH

      // Note: encrypted msgs over direct USB I/O are not yet supported.
      bytes = write( pmctl->st.usbdio.usbdiofd, p, msg_len );

      if ( bytes < 0 )  // error
      {
        rc = mbg_get_last_error( "write failed for USB direct I/O" );
        goto out;
      }

      if ( bytes != msg_len )
        goto out_write_failed;

    } break;
  #endif

    default:
      rc = MBG_ERR_NOT_SUPP_ON_OS;
      goto out;

  }  // switch

  rc = MBG_SUCCESS;
  goto out;

out_write_failed:
  rc = MBG_ERR_BYTES_WRITTEN;

out:
  return rc;

}  /* xmt_tbuff */



/*HDR*/
/**
 * @brief Send a binary command message without parameters
 *
 * This function doesn't protect access to the device by a mutex,
 * so the calling function has to take care of the mutex handling,
 * or ::mbgextio_xmt_cmd which does so should be called preferably
 * from applications.
 *
 * @param[in,out] pmctl   Pointer to a valid message control structure
 * @param[in]     p_addr  Pointer to XBP address specifier
 * @param[in]     cmd     One of the command codes enumerated in ::GPS_CMD_CODES
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_xmt_cmd
 * @see ::mbgextio_xmt_cmd_us
 * @see ::xmt_cmd_us
 */
int xmt_cmd( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, GPS_CMD cmd )
{
  MBG_MSG_BUFF *pmb = pmctl->xmt.pmb;

  pmb->hdr.len = 0;
  pmb->hdr.cmd = cmd;

  return xmt_tbuff( pmctl, p_addr );

}  /* xmt_cmd */



/*HDR*/
/**
 * @brief Send a binary command message with ushort (16 bit) parameter
 *
 * This function doesn't protect access to the device by a mutex,
 * so the calling function has to take care of the mutex handling,
 * or ::mbgextio_xmt_cmd_us which does so should be called preferably
 * from applications.
 *
 * @param[in,out] pmctl   Pointer to a valid message control structure
 * @param[in]     p_addr  Pointer to XBP address specifier
 * @param[in]     cmd     One of the command codes enumerated in ::GPS_CMD_CODES
 * @param[in]     us      The 16 bit message parameter (data)
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_xmt_cmd_us
 * @see ::mbgextio_xmt_cmd
 * @see ::xmt_cmd
 */
int xmt_cmd_us( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, GPS_CMD cmd, uint16_t us )
{
  MBG_MSG_BUFF *pmb = pmctl->xmt.pmb;

  pmb->u.msg_data.us = us;
  pmb->hdr.len = sizeof( pmb->u.msg_data.us );
  pmb->hdr.cmd = cmd;

  return xmt_tbuff( pmctl, p_addr );

}  /* xmt_cmd_us  */



/*HDR*/
/**
 * @brief Init reception of a binary message
 *
 * @param[in,out] prctl  Pointer to a valid receive control structure
 *
 * @see ::check_transfer
 */
void init_transfer( MBG_MSG_RCV_CTL *prctl )
{
  memset( &prctl->rcv_state, 0, sizeof( prctl->rcv_state ) );
  prctl->rcv_state.cur = (uint8_t *) prctl->pmb;   // point to start of buffer

}  // init_transfer



/*HDR*/
/**
 * @brief Check an incoming data stream for a binary message
 *
 * Check the sequence of incoming characters for blocks of
 * binary message data. Binary messages are saved in a buffer
 * which is part of the ::MBG_MSG_RCV_CTL structure and the
 * caller checks the return value to get the receive status.
 *
 * This function doesn't protect access to the device by a mutex,
 * so the calling function has to take care of the mutex handling.
 *
 * @param[in,out] prctl  Pointer to a valid receive control structure
 * @param[in]     c      A byte from the incoming data stream
 *
 * @return One of the ::TR_STATUS_CODES
 */
int check_transfer( MBG_MSG_RCV_CTL *prctl, uint8_t c )
{
  MBG_MSG_BUFF *pmb = prctl->pmb;
  MSG_HDR *pmh = &pmb->hdr;

  if ( prctl->rcv_state.cnt == 0 )             /* not receiving yet */
  {
    if ( c != START_OF_HEADER )
      return TR_WAITING;             /* ignore this character */

    /* initialize receiving */
    init_transfer( prctl );
    mbg_tmo_get_time( &prctl->rcv_state.tstamp );
    prctl->rcv_state.cnt = sizeof( *pmh );     /* prepare to rcv msg header */

    return TR_RECEIVING;
  }


  /* SOH has already been received */

  if ( prctl->rcv_state.cur < ( (uint8_t *) prctl->pmb + sizeof( *prctl->pmb ) ) )
  {
    *prctl->rcv_state.cur = c;          /* save incoming character */
    prctl->rcv_state.cur++;
  }
  else                        /* don't write beyond buffer */
    prctl->rcv_state.flags |= MBG_MSG_RCV_CTL_OVERFLOW;


  prctl->rcv_state.cnt--;

  if ( prctl->rcv_state.cnt )           /* transfer not complete */
    return TR_RECEIVING;


  /* cnt == 0, so the header or the whole message is complete */

  if ( !( prctl->rcv_state.flags & MBG_MSG_RCV_CTL_RCVD_HDR ) )  /* header complete now */
  {
    unsigned int data_len;

    if ( chk_hdr_csum( pmh ) < 0 )  /* error */
    {
      init_transfer( prctl );       /* restart receiving */
      return TR_CSUM_HDR;           /* invalid header checksum received */
    }

    if ( pmh->len == 0 )            /* no data to wait for */
      goto msg_complete;            /* message complete */

    data_len = pmh->len;

    prctl->rcv_state.cnt = data_len;     /* save number of bytes to wait for */
    prctl->rcv_state.flags |= MBG_MSG_RCV_CTL_RCVD_HDR;  /* flag header complete */

    if ( data_len > ( prctl->buf_size - sizeof( *pmh ) ) )
      prctl->rcv_state.flags |= MBG_MSG_RCV_CTL_MSG_TOO_LONG;

    return TR_RECEIVING;
  }

  if ( prctl->rcv_state.flags & MBG_MSG_RCV_CTL_OVERFLOW )
    return TR_OVERFLOW;

  /* Header and data have been received. The header checksum has been */
  /* checked, now recompute and compare data checksum. */

  if ( chk_data_csum( pmb ) < 0 )   /* error */
  {
    init_transfer( prctl );         /* restart receiving */
    return TR_CSUM_DATA;            /* invalid header checksum received */
  }

msg_complete:
  return TR_COMPLETE;               /* message complete, must be evaluated */

}  /* check_transfer */



