
/**************************************************************************
 *
 *  $Id: mbg_lx.h 1.18 2024/12/10 14:47:39Z martin.burnicki REL_M $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    OS dependent definitions/redefinitions for Linux.
 *
 * -----------------------------------------------------------------------
 *  $Log: mbg_lx.h $
 *  Revision 1.18  2024/12/10 14:47:39Z  martin.burnicki
 *  Consideration of kernel API backports in RHEL 9.4/9.5.
 *  Revision 1.17  2024/11/26 15:49:29Z  martin.burnicki
 *  New preprocessor symbol KERNEL_HAS_LINUX_UNALIGNED_H.
 *  Revision 1.16  2024/11/26 14:14:24Z  martin.burnicki
 *  Reorderd entries and use more common symbol names.
 *  Revision 1.15  2024/07/23 14:42:01Z  martin.burnicki
 *  Fixed some potential portability problems, cleaned up
 *  and removed some obsolete code.
 *  Revision 1.14  2021/03/22 23:21:23  martin
 *  Updated some comments.
 *  Revision 1.13  2018/04/12 09:46:12  martin
 *  Code to check if the kernel supports testing if a buffer
 *  is DMA capable.
 *  Revision 1.12  2017/07/05 16:54:16  martin
 *  Removed some obsolete code.
 *  Revision 1.11  2013/03/04 15:43:49  martin
 *  Account for __devinit and friends removed around kernel 3.8.
 *  Revision 1.10  2013/02/19 15:56:15  martin
 *  Moved more kernel version compatibility checks here.
 *  Revision 1.9  2012/10/12 11:23:11  martin
 *  Don't include linux/autoconf.h anymore.
 *  Revision 1.8  2009/03/13 09:04:18  martin
 *  Added default definition for _PCPS_USB_FULL_CYCLIC_INTV.
 *  Revision 1.7  2008/12/19 15:22:05  martin
 *  Moved lots of code here which provides compatibility across
 *  kernel versions.
 *  Added preliminary check if kthread API can be used. However,
 *  the exact version number still needs to be verified.
 *  Added a check whether USB timeouts have to be specified
 *  in jiffies (old) or milliseconds (newer).
 *  Revision 1.6  2007/02/22 15:26:56  martin
 *  Include linux/autoconf.h rather than linux/config.h since the latter has been
 *  a stub for autoconf.h which has been removed around kernel version 2.6.19.
 *  Revision 1.5  2006/09/20 12:09:29  martin
 *  Removed port I/O function redefinitions which have been replaced
 *  by generic function definitions in mbggenio.h.
 *  Revision 1.4  2006/07/07 12:06:40  martin
 *  Include modversions.h if required by the kernel.
 *  Revision 1.3  2001/05/31 09:21:09  martin
 *  #include <asm/io.h> instead of <sys/io.h>
 *  Revision 1.2  2001/03/05 16:36:11  MARTIN
 *  Renamed file.
 *
 **************************************************************************/

#ifndef _MBG_LX_H
#define _MBG_LX_H

#include <linux/version.h>

#if !defined( KERNEL_VERSION ) || ( LINUX_VERSION_CODE < KERNEL_VERSION( 2, 6, 0 ) )
  #error "Linux kernel 2.6 or newer is required!"
#endif


#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/interrupt.h>
#include <linux/proc_fs.h>
#include <linux/init.h>
#include <linux/param.h>

#include <asm/uaccess.h>
#include <asm/io.h>


// For compatibility with other target platforms.
#define _MBG_INIT_CODE_ATTR           __init


// RedHat Enterprise Linux (RHEL) kernels usually
// provide some definitions that can be used to
// track RHEL-specific kernel API changes.
// For non-RHEL we provide some dummy declarations.

#if !defined( RHEL_RELEASE_CODE )
  #define RHEL_RELEASE_CODE  0
#endif

#if !defined( RHEL_RELEASE_VERSION )
  #define RHEL_RELEASE_VERSION(a,b) (((a) << 8) + (b))
#endif


// Definitions for clock() support in kernel drivers.
extern unsigned long volatile jiffies;
#define clock() jiffies


#if !defined( IRQ_RETVAL )
  // Defines for compatibility with very old kernels,
  // probably before 2.4.

  typedef void irqreturn_t;
  #define IRQ_NONE
  #define IRQ_HANDLED
  // #define IRQ_RETVAL(x)
#endif


#if !defined( NEW_WAIT_QUEUE )
  // New wait queue types and macros have been defined around kernel 2.2.18.

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 2, 18 ) )
    #define NEW_WAIT_QUEUE  1
  #else
    #define NEW_WAIT_QUEUE  0
  #endif
#endif


#if !defined( NEW_FASYNC )
  // A third parameter to kill_fasync has been added in kernel 2.3.21,
  // and is *not* present in *stock* 2.2.14 and 2.2.15. However, some
  // distros patch 2.2.x kernels to add this parameter.

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 2, 18 ) )
    #define NEW_FASYNC  1
  #else
    #define NEW_FASYNC  0
  #endif
#endif


#if !defined( NEED_IMINOR )
  // TODO Version needs verification !

  #if ( LINUX_VERSION_CODE < KERNEL_VERSION( 2, 4, 0 ) )
    #define NEED_IMINOR  1
  #else
    #define NEED_IMINOR  0
  #endif
#endif

#if NEED_IMINOR
  #define iminor( _d ) MINOR( (_d)->i_rdev )
#endif


#if !defined( _DEFAULT_MBG_TGT_LINUX_USE_PCI_PNP )
  // First kernel version for which we support PCI by default.

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 4, 0 ) )
    #define _DEFAULT_MBG_TGT_LINUX_USE_PCI_PNP  1
  #else
    #define _DEFAULT_MBG_TGT_LINUX_USE_PCI_PNP  0
  #endif
#endif


#if !defined( NEW_FASYNC2 )
  // Type of first parameter of kill_fasync() changed
  // in kernel 2.4.0-test2.

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 4, 0 ) )
    #define NEW_FASYNC2  1
  #else
    #define NEW_FASYNC2  1
  #endif
#endif


#if !defined( KERNEL_FILE_OPS_HAVE_OWNER )
  // A field "owner" that needs to be initialized properly
  // has been added to struct file_operations in kernel 2.4.0.

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 4, 0 ) )
    #define KERNEL_FILE_OPS_HAVE_OWNER  1
  #else
    #define KERNEL_FILE_OPS_HAVE_OWNER  0
  #endif
#endif


#if !defined( KERNEL_HAS_WAIT_EVENT )
  // Function wait_event_interruptible() which replaces the
  // older interruptible_sleep_on() function has been added
  // in kernel 2.4.20.

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 4, 20 ) )
    #define KERNEL_HAS_WAIT_EVENT  1
  #else
    #define KERNEL_HAS_WAIT_EVENT  0
  #endif
#endif


#if !defined( KERNEL_USES_DEVFS )
  // devfs was introduced in kernel ~2.4 to dynamically
  // create device nodes. Starting with kernel ~2.6, it was
  // obsoleted by the udev system which runs in user space.
  // We support it if the associated kernel config parameter
  // is defined.

  #if defined( CONFIG_DEVFS_FS )
    #define KERNEL_USES_DEVFS 1
  #else
    #define KERNEL_USES_DEVFS 0
  #endif
#endif


#if !defined( _PCPS_MUST_UPDATE_USE_COUNT )
  // Before kernel 2.6.0, drivers have to update
  // their use count by themselves.

  #if ( LINUX_VERSION_CODE < KERNEL_VERSION( 2, 6, 0 ) )
    #define _PCPS_MUST_UPDATE_USE_COUNT  1
  #else
    #define _PCPS_MUST_UPDATE_USE_COUNT  0
  #endif
#endif


#if !defined( _DEFAULT_MBG_TGT_LINUX_USE_USB )
  // First kernel version for which we support USB by default.

  #if ( ( defined( CONFIG_USB ) || defined( CONFIG_USB_MODULE ) ) && \
      ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 0 ) ) )
    #define _DEFAULT_MBG_TGT_LINUX_USE_USB   1
  #else
    #define _DEFAULT_MBG_TGT_LINUX_USE_USB   0
  #endif
#endif


#if !defined( KERNEL_HAS_LINUX_CLASS )
  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 0 ) )
    #define KERNEL_HAS_LINUX_CLASS  1
  #else
    #define KERNEL_HAS_LINUX_CLASS  0
  #endif
#endif

#if KERNEL_HAS_LINUX_CLASS
  #include <linux/cdev.h>
#endif


#if !defined( KERNEL_VMA_HAS_VM_PGOFF )
  // Determine whether struct vm_area_struct has a vm_pgoff field.
  // TODO Version needs to be verified.

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 0 ) )
    #define KERNEL_VMA_HAS_VM_PGOFF  1
  #else
    #define KERNEL_VMA_HAS_VM_PGOFF  0
  #endif
#endif


#if !defined( STRUCT_USB_DEVICE_HAS_STATIC_SERIAL )
  #define STRUCT_USB_DEVICE_HAS_STATIC_SERIAL  0
  // There seems to be a patch in SuSE Linux 9.1 / 2.6.4-52
  // which supports this ...
#endif


#if !defined( KERNEL_ESPIPE_BY_VFS )
  // ESPIPE logic has been removed from drivers and is
  // now handled by the VFS layer.
  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 8 ) )
    #define KERNEL_ESPIPE_BY_VFS  1
  #else
    #define KERNEL_ESPIPE_BY_VFS  0
  #endif
#endif


#if !defined( KERNEL_HAS_REMAP_PFN )
  // remap_pfn_range() and io_remap_pfn_range() were
  // introduced around 2.6.10 to replace remap_page_range()
  // and io_remap_page_range(), but in the next kernel
  // versions the old functions are still supported,
  // so it does not seem critical at which kernel version
  // exactly we start using the new functions.
  //
  // TODO Something like this could be used in the Makefile:
  //
  //  VMA_PARAM_IN_REMAP=`grep remap_page_range
  //  $PATH_LINUX_INCLUDE/linux/mm.h|grep vma`
  //  if [ -z "$VMA_PARAM_IN_REMAP" ]; then
  //   export REMAP_PAGE_RANGE_PARAM="4"
  //  else
  //   export REMAP_PAGE_RANGE_PARAM="5"
  //  endif

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 10 ) )
    #define KERNEL_HAS_REMAP_PFN  1
  #else
    #define KERNEL_HAS_REMAP_PFN  0
  #endif
#endif


#if !defined( KERNEL_HAS_KTHREAD )
  // Check if kernel threads are supported.
  // TODO Is version 2.6.10 correct?

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 10 ) )
    #define KERNEL_HAS_KTHREAD  1
  #else
    #define KERNEL_HAS_KTHREAD  0
  #endif
#endif

#if KERNEL_HAS_KTHREAD
  #include <linux/kthread.h>
#endif


#if !defined( KERNEL_HAS_UNLOCKED_IOCTL )
  // The 'struct file_operations' contains the fields
  // 'unlocked_ioctl' and 'compat_ioctl' since kernel ~2.6.11,
  // and the symbols HAVE_UNLOCKED_IOCTL and HAVE_COMPAT_IOCTL
  // can be used to determine if they are supported, or the
  // legacy IOCTL has to be used. However, these symbols
  // were removed in kernel 5.9, so we need a workaround.

  #if defined( HAVE_UNLOCKED_IOCTL ) || ( LINUX_VERSION_CODE >= KERNEL_VERSION( 3, 0, 0 ) )
    #define KERNEL_HAS_UNLOCKED_IOCTL  1
  #else
    #define KERNEL_HAS_UNLOCKED_IOCTL  0
  #endif
#endif

#if !defined( KERNEL_HAS_COMPAT_IOCTL )
  // Same as for KERNEL_HAS_UNLOCKED_IOCTL, above.
  #define KERNEL_HAS_COMPAT_IOCTL  KERNEL_HAS_UNLOCKED_IOCTL
#endif


#if !defined( _PCPS_USB_FULL_CYCLIC_INTV )
  // Early Linux 2.6.x kernels generate a syslog msg on each USB timeout,
  // so the syslog would be flooded with USB timeout msgs if we used
  // short timeouts with retries.

  #if ( LINUX_VERSION_CODE < KERNEL_VERSION( 2, 6, 11 ) )
    #define _PCPS_USB_FULL_CYCLIC_INTV  1
  #else
    #define _PCPS_USB_FULL_CYCLIC_INTV  0
  #endif
#endif


#if !defined( KERNEL_USB_TIMEOUT_IN_MS )
  // Starting at some version 2.6.11 or 2.6.12, timeout values
  // for the USB functions have to be specified in ms, older
  // kernels expect USB timeout intervals in jiffies.

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 12 ) )
    #define KERNEL_USB_TIMEOUT_IN_MS  1
  #else
    #define KERNEL_USB_TIMEOUT_IN_MS  0
  #endif
#endif

#if !KERNEL_USB_TIMEOUT_IN_MS
  // For older Linux kernels that expect USB timeout intervals
  // in jiffies, we specify a conversion macro.
  #define _pcps_ms_to_usb_timeout( _ms )  ( (_ms) * HZ / 1000 )
#endif


#if !defined( STRUCT_USB_DEVICE_HAS_SERIAL )
  // struct usb has a field for the serial number.

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 12 ) )
    #define STRUCT_USB_DEVICE_HAS_SERIAL  1
  #else
    #define STRUCT_USB_DEVICE_HAS_SERIAL  0
  #endif
#endif


#if !defined( KERNEL_HAS_LINUX_CLASS_SIMPLE )
  #if ( KERNEL_HAS_LINUX_CLASS && ( LINUX_VERSION_CODE < KERNEL_VERSION( 2, 6, 13 ) ) )
    #define KERNEL_HAS_LINUX_CLASS_SIMPLE  1
  #else
    #define KERNEL_HAS_LINUX_CLASS_SIMPLE  0
  #endif
#endif


#if !defined( KERNEL_HAS_LINUX_CLASS_CREATE )
  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 13 ) )
    #define KERNEL_HAS_LINUX_CLASS_CREATE  1
  #else
    #define KERNEL_HAS_LINUX_CLASS_CREATE  0
  #endif
#endif


#if !defined( KERNEL_CLASS_DEVICE_CREATE_WITH_PARENT )
  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 14 ) )
    #define KERNEL_CLASS_DEVICE_CREATE_WITH_PARENT  1
  #else
    #define KERNEL_CLASS_DEVICE_CREATE_WITH_PARENT  0
  #endif
#endif


#if !defined( KERNEL_FLUSH_WITH_LOCK_OWNER_ID )
  // Starting with kernel 2.6.18, the prototype for the
  // flush function in struct file_operations in fs.h
  // has been modified to take the POSIX lock owner ID
  // as additional parameter.
  // The same patch also defined FL_CLOSE in fs.h, so
  // we additionally check whether this symbol is defined.
  // At least for SUSE SLES10 this has been backported
  // to a kernel which reports to be 2.6.16.

  #if ( ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 18 ) ) \
        || defined( FL_CLOSE ) )
    #define KERNEL_FLUSH_WITH_LOCK_OWNER_ID  1
  #else
    #define KERNEL_FLUSH_WITH_LOCK_OWNER_ID  0
  #endif
#endif


#if !defined( KERNEL_REQUEST_IRQ_WO_REGS )
  // The prototype of the interrupt handler function has
  // changed around 2.6.19. The newer version does not take
  // a pointer to a register set. This also seems to include
  // linux/bottom_half.h from linux/interrupt.h, so a criteria
  // for this is also if _LINUX_BH_H is defined after
  // linux/interrupt.h has been included.

  #if ( ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 19 ) \
        || defined( _LINUX_BH_H ) ) )
    #define KERNEL_REQUEST_IRQ_WO_REGS  1
  #else
    #define KERNEL_REQUEST_IRQ_WO_REGS  0
  #endif
#endif


#if !defined( KERNEL_CLASS_DEV_OBSOLETED )
  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 26 ) )
    #define KERNEL_CLASS_DEV_OBSOLETED  1
  #else
    #define KERNEL_CLASS_DEV_OBSOLETED  0
  #endif
#endif


#if !defined( KERNEL_HAS_LINUX_SEMAHORE_H )
  // Have to include <linux/semaphore.h>
  // instead of <asm/semaphore.h>

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 26 ) )
    #define KERNEL_HAS_LINUX_SEMAHORE_H  1
  #else
    #define KERNEL_HAS_LINUX_SEMAHORE_H  0
  #endif
#endif


#if !defined( KERNEL_DEVICE_CREATE_WITH_DEV_DATA )
  // create_device() takes a pointer to private driver data
  // useful fo callbacks.

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 27 ) )
    #define KERNEL_DEVICE_CREATE_WITH_DEV_DATA  1
  #else
    #define KERNEL_DEVICE_CREATE_WITH_DEV_DATA  0
  #endif
#endif


#if !defined( KERNEL_SUPP_GENERIC_GPIO )
  // TODO Exact kernel version to be determined!
  // At least 2.6.34 provides linux/gpio.h.

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 34 ) )
    #define KERNEL_SUPP_GENERIC_GPIO  1
  #else
    #define KERNEL_SUPP_GENERIC_GPIO  0
  #endif
#endif


// Starting around kernel 3.8, __devinit and friends
// are not supported anymore, so we provide empty definitions.

#if !defined( __devinit )
  #define __devinit
#endif

#if !defined( __devinitdata )
  #define __devinitdata
#endif

#if !defined( __devexit )
  #define __devexit
#endif


// Early hotplug systems may have defined __devexit,
// but not __devexit_p, and similarly, kernel 3.8 and newer
// may not have defined __devexit_p anymore.

#if !defined( __devexit_p )
  #define __devexit_p( _x ) _x
#endif


#if !defined( _PCPS_CHK_BUFFER_DMA_CAPABLE )
  // Starting with kernel 4.9 we *have to* make sure that e.g.
  // USB I/O buffers are DMA capable. Otherwise the kernel will
  // generate a trap-like message and prevent the driver from
  // working, even if DMA is not used by the driver.
  // The inline functions is_vmalloc_addr() and object_is_on_stack()
  // can be used to check this, but these functions may not be
  // available in older 2.6 kernels.
  // is_vmalloc_addr() is supported in kernels ~2.6.25-rc1 and later,
  // object_is_on_stack() is supported by ~2.6.27-rc1 and later,
  // so we implement this for kernels which provide at least both
  // of these functions.

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 2, 6, 27 ) )
    #define _PCPS_CHK_BUFFER_DMA_CAPABLE  1
  #else
    #define _PCPS_CHK_BUFFER_DMA_CAPABLE  0
  #endif
#endif

#if _PCPS_CHK_BUFFER_DMA_CAPABLE
  // Since kernel 4.11, the inline function object_is_on_stack() has
  // been moved from linux/sched.h to linux/sched/task_stack.h.

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 4, 11, 0 ) )
    #include <linux/sched/task_stack.h>
  #else
    #include <linux/sched.h>
  #endif

  #define _pcps_buffer_is_dma_capable( _b ) \
    ( !is_vmalloc_addr( _b ) && !object_is_on_stack( _b ) )
#endif


#if !defined( KERNEL_HAS_VMA_VM_FLAGS_WRAPPER_FUNCTIONS )
  // Commit bc292ab00f6c of the kernel source introduced
  // vma->vm_flags wrapper functions to avoid race conditions
  // when the flags are updated. See the discussion at
  // https://lore.kernel.org/lkml/ZA7x9y60sfGOanHl@kroah.com/T/
  // These wrapper functions first appeared in v6.3-rc1~113^2~116.
  //
  // Backported to RHEL 9.5 with kernel 5.14.0-503.11.1.el9_5,
  // centos-stream-9 commit a8950d4008117328fc0f7d097645d1e582cbb166.
  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 6, 3, 0 ) ) || \
      ( RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION( 9, 5 ) )
    #define KERNEL_HAS_VMA_VM_FLAGS_WRAPPER_FUNCTIONS  1
  #else
    #define KERNEL_HAS_VMA_VM_FLAGS_WRAPPER_FUNCTIONS  0
  #endif
#endif


#if !defined( KERNEL_CLASS_WITHOUT_OWNER )
  // Commit 6e30a66433afee90e902ced95d7136e8f7edcc7e of the
  // kernel source removed the struct module *owner parameter
  // from the __class_register() function. See the discussion at
  // https://lore.kernel.org/lkml/2023041123-tractor-quake-c44d@gregkh/
  // The change first appeared in v6.4-rc1~115^2~102.
  //
  // Backported to RHEL 9.4 with kernel 5.14.0-427.13.1.el9_4,
  // centos-stream-9 commit 094c7e28221ff336f12c8e2ba8433789f624487d.
  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 6, 4, 0 ) ) || \
      ( RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION( 9, 4 ) )
    #define KERNEL_CLASS_WITHOUT_OWNER  1
  #else
    #define KERNEL_CLASS_WITHOUT_OWNER  0
  #endif
#endif


#if !defined( KERNEL_HAS_LINUX_UNALIGNED_H )
  // Commit 5f60d5f6bbc12e782fac78110b0ee62698f3b576 of the
  // kernel source moved asm/unaligned.h to linux/unaligned.h.
  // The change first appeared in v6.12-rc2~37^2.

  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION( 6, 12, 0 ) )
    #define KERNEL_HAS_LINUX_UNALIGNED_H  1
  #else
    #define KERNEL_HAS_LINUX_UNALIGNED_H  0
  #endif
#endif


#endif  /* _MBG_LX_H */
