
/**************************************************************************
 *
 *  $Id: str_ext_sys_info.c 1.1 2025/05/16 11:22:49Z martin.burnicki REL_M $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Utility functions for the MBG_EXT_SYS_INFO structure.
 *
 * -----------------------------------------------------------------------
 *  $Log: str_ext_sys_info.c $
 *  Revision 1.1  2025/05/16 11:22:49Z  martin.burnicki
 *  Initial revision.
 *
 **************************************************************************/

#define _STR_EXT_SYS_INFO
  #include <str_ext_sys_info.h>
#undef _STR_EXT_SYS_INFO

#if defined( MBG_TGT_KERNEL )
  #include <mbgddmsg.h>
#else
  #include <str_util.h>
  #include <stdio.h>
#endif



#if !defined( MBG_TGT_KERNEL )

#define mbg_kdd_snprintf snprintf_safe

__attribute__( ( format( printf, 2, 3 ) ) )
void mbg_kdd_msg( int lvl, const char *fmt, ... )
{
  va_list args;

  va_start( args, fmt );
  vprintf( fmt, args );
  printf( "\n" );
  va_end( args );

}  // mbg_kdd_msg

#endif


static const char *ext_sys_os_type_names[N_MBG_EXT_SYS_INFO_OS_TYPES] = MBG_EXT_SYS_INFO_OS_STRS;
static const char *ext_sys_proc_type_names[N_MBG_EXT_SYS_INFO_PROC_TYPES] = MBG_EXT_SYS_INFO_PROC_STRS;
static const char *ext_sys_fpga_type_names[N_MBG_EXT_SYS_INFO_FPGA_TYPES] = MBG_EXT_SYS_INFO_FPGA_STRS;
static const char *ext_sys_core_mod_type_names[N_MBG_EXT_SYS_INFO_CORE_MOD_TYPES] = MBG_EXT_SYS_INFO_CORE_MOD_STRS;
static const char str_spc_no_bit[] = " (bit not set)";
static const char str_unkn_braced[] = "(unknown)";


/*HDR*/
int snprint_check_bit_set( char *s, size_t max_len, bool must_print )
{
  int n = 0;

  if ( must_print )
    n += mbg_kdd_snprintf( s, max_len, "%s", str_spc_no_bit );

  return n;

}  // snprint_check_bit_set



/*HDR*/
int snprint_ext_sys_rev_info( char *s, size_t max_len, uint32_t rev_num,
                              bool verbose )
{
  int major;
  int minor;
  int patch;
  int n;

  _mbg_decode_revision( rev_num, major, minor, patch );

  n = mbg_kdd_snprintf( s, max_len, "%d.%02d", major, minor );

  if ( patch || verbose )
    n += mbg_kdd_snprintf( &s[n], max_len - n, ".%d", patch );

  return n;

}  // snprint_ext_sys_rev_info



/*HDR*/
int snprint_ext_sys_hw_rev_info( char *s, size_t max_len, const MBG_EXT_SYS_INFO *p,
                                 bool verbose )
{
  int n = snprint_ext_sys_rev_info( s, max_len, p->hw_rev, verbose );

  n += snprint_check_bit_set( &s[n], max_len - n,
           p->hw_rev && !( p->supp_members & MBG_EXT_SYS_INFO_MSK_HW_REV ) );

  return n;

}  // snprint_ext_sys_hw_rev_info



/*HDR*/
int snprint_ext_sys_sw_rev_info( char *s, size_t max_len, const MBG_EXT_SYS_INFO *p,
                                 bool verbose )
{
  int n = snprint_ext_sys_rev_info( s, max_len, p->sw_rev, verbose );

  n += snprint_check_bit_set( &s[n], max_len - n,
           p->hw_rev && !( p->supp_members & MBG_EXT_SYS_INFO_MSK_HW_REV ) );

  return n;

}  // snprint_ext_sys_sw_rev_info



/*HDR*/
int snprint_ext_sys_os_type_info( char *s, size_t max_len, const MBG_EXT_SYS_INFO *p,
                                  bool verbose )
{
  int n;

  const char *cp = ( p->mbg_os_type < N_MBG_EXT_SYS_INFO_OS_TYPES ) ?
                     ext_sys_os_type_names[p->mbg_os_type] : str_unkn_braced;

  n = mbg_kdd_snprintf( s, max_len, "%s", cp );

  n += snprint_check_bit_set( &s[n], max_len - n,
           p->hw_rev && !( p->supp_members & MBG_EXT_SYS_INFO_MSK_OS_TYPE ) );

  return n;

}  // snprint_ext_sys_os_type_info



/*HDR*/
int snprint_ext_sys_fpga_info( char *s, size_t max_len, const MBG_EXT_SYS_INFO *p,
                               bool verbose )
{
  int n;

  const char *cp = ( p->fpga_type < N_MBG_EXT_SYS_INFO_FPGA_TYPES ) ?
                     ext_sys_fpga_type_names[p->fpga_type] : str_unkn_braced;

  n = mbg_kdd_snprintf( s, max_len, "%s", cp );

  if ( p->fpga_rev || ( p->supp_members & MBG_EXT_SYS_INFO_MSK_FPGA_REV ) || verbose )
  {
    n += mbg_kdd_snprintf( &s[n], max_len - n, " (" );
    n += snprint_ext_sys_rev_info( &s[n], max_len - n, p->fpga_rev, verbose );
    n += mbg_kdd_snprintf( &s[n], max_len - n, ")" );
  }

  n += snprint_check_bit_set( &s[n], max_len - n,
           p->fpga_rev && !( p->supp_members & MBG_EXT_SYS_INFO_MSK_FPGA_REV ) );

  return n;

}  // snprint_ext_sys_fpga_info



/*HDR*/
int snprint_ext_sys_core_module_info( char *s, size_t max_len, const MBG_EXT_SYS_INFO *p,
                                      bool verbose )
{
  int n;

  const char *cp = ( p->core_mod_type < N_MBG_EXT_SYS_INFO_CORE_MOD_TYPES ) ?
                     ext_sys_core_mod_type_names[p->core_mod_type] : str_unkn_braced;

  n = mbg_kdd_snprintf( s, max_len, "%s", cp );

  if ( p->core_mod_rev || ( p->supp_members & MBG_EXT_SYS_INFO_MSK_CORE_MOD_REV ) || verbose )
  {
    n += mbg_kdd_snprintf( &s[n], max_len - n, " (" );
    n += snprint_ext_sys_rev_info( &s[n], max_len - n, p->core_mod_rev, verbose );
    n += mbg_kdd_snprintf( &s[n], max_len - n, ")" );
  }

  n += snprint_check_bit_set( &s[n], max_len - n,
           p->core_mod_rev && !( p->supp_members & MBG_EXT_SYS_INFO_MSK_CORE_MOD_REV )  );

  return n;

}  // snprint_ext_sys_core_module_info



/*HDR*/
void dump_ext_sys_info( int log_lvl, const MBG_EXT_SYS_INFO *p, const char *info, int verbosity_lvl )
{
  char ws[128];
  const char *fmt = "  %-12s: %s";
  bool verbose = verbosity_lvl > 0;
  const char *cp;

  if ( info )
    mbg_kdd_msg( log_lvl, "%s", info );

  if ( verbose )
  {
    mbg_kdd_snprintf( ws, sizeof( ws ), "0x%08lX", (ulong) p->supp_members );
    mbg_kdd_msg( log_lvl, fmt, "Supp. fields", ws );
  }

  if ( ( p->supp_members & MBG_EXT_SYS_INFO_MSK_HW_REV ) || verbose )
  {
    snprint_ext_sys_hw_rev_info( ws, sizeof( ws ), p, verbose );
    mbg_kdd_msg( log_lvl, fmt, "HW revision", ws );
  }

  if ( ( p->supp_members & MBG_EXT_SYS_INFO_MSK_SW_REV ) || verbose )
  {
    snprint_ext_sys_sw_rev_info( ws, sizeof( ws ), p, verbose );
    mbg_kdd_msg( log_lvl, fmt, "SW revision", ws );
  }

  if ( ( p->supp_members & MBG_EXT_SYS_INFO_MSK_OS_TYPE ) || verbose )
  {
    snprint_ext_sys_os_type_info( ws, sizeof( ws ), p, verbose );
    mbg_kdd_msg( log_lvl, fmt, "OS type", ws );
  }

  if ( p->proc_type || verbose )
  {
    cp = ( p->proc_type < N_MBG_EXT_SYS_INFO_PROC_TYPES) ?
           ext_sys_proc_type_names[p->proc_type] : str_unkn_braced;
    mbg_kdd_msg( log_lvl, fmt, "Processor", cp );
  }

  if ( p->fpga_type || ( p->supp_members & MBG_EXT_SYS_INFO_MSK_FPGA_REV ) || verbose )
  {
    snprint_ext_sys_fpga_info( ws, sizeof( ws ), p, verbose );
    mbg_kdd_msg( log_lvl, fmt, "FPGA", ws );
  }

  if ( p->core_mod_type || ( p->supp_members & MBG_EXT_SYS_INFO_MSK_CORE_MOD_REV ) || verbose )
  {
    snprint_ext_sys_core_module_info( ws, sizeof( ws ), p, verbose );
    mbg_kdd_msg( log_lvl, fmt, "Core Module", ws );
  }

}  // dump_ext_sys_info

