
/**************************************************************************
 *
 *  $Id: mbgcfg_util.c 1.3 2025/07/25 14:47:22Z martin.burnicki REL_M $
 *
 *  Common helper functions for Meinberg configuration programs.
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 * -----------------------------------------------------------------------
 *  $Log: mbgcfg_util.c $
 *  Revision 1.3  2025/07/25 14:47:22Z  martin.burnicki
 *  Better control to which channel output is printed.
 *  Revision 1.2  2025/07/25 13:19:22Z  martin.burnicki
 *  Made function print_bit_mask_list() more versatile.
 *  Revision 1.1  2025/07/10 14:50:35Z  martin.burnicki
 *  Initial revision with code from mbgtools-lx/mbgctrl.
 *
 **************************************************************************/

#define _MBGCFG_UTIL
  #include <mbgcfg_util.h>
#undef _MBGCFG_UTIL

#include <str_util.h>

#include <ctype.h>


#define DEBUG_MBGCFG_UTIL  0


#if 0   // FIXME TODO Do we need this?

/*HDR*/
int snprintf_spaces( char *s, size_t max_len, int n_spaces )
{
  int n = snprintf_safe( s, max_len, "%*s", n_spaces, mbgcfg_str_empty );

  return n;

}  // snprintf_spaces

#endif


/*HDR*/
/**
 * @brief Print a specific error message.
 *
 * @param[in] info   Message info to be printed
 *
 * @return Always ::MBG_SUCCESS
 */
int err_msg( const char *info )
{
  fprintf( stderr, "This device %s.\n", info );

  return MBG_SUCCESS;

}  // err_msg

MBG_ERR_MSG_FNC err_msg;



/*HDR*/
long check_range( long l, long min_limit, long max_limit )
{
  if ( l < min_limit )
    return min_limit;

  if ( l > max_limit )
    return max_limit;

  return l;

}  // check_range



/*HDR*/
/**
 * @brief Print a formatted list of options defined by a bit mask.
 *
 * @param[in]  fp          Fhe @a FILE to which to print, e.g. stdout or stderr.
*
 * @param[in]  info_1      Optional string specifying the kind of information, may be @a NULL.
 * @param[in]  info_2      Optional string with extended information, may be @a NULL.
 *
 * @param[in]  supp_mask   Bit mask of supported options from the option list.
 *                         If @ref CTRL_PRINT_ALL is set in @p ctrl_flags, the original
 *                         value of @p supp_mask is ignored, and the mask is extended
 *                         to support all @p n_known options.
 *
 * @param[in]  n_known     Number of known values in the option list.
 *
 * @param[in]  names       Table of option names with at least @p n_known entries, may be @a NULL.
 *                         If no table is provided, @p s_func is called instead, if specified.
 *
 * @param[in]  s_fnc       Optional function that retrieves a string, may be @a NULL.
 * @param[in]  inst_idx    Parameter for the optional function specified in @p s_fnc.
 *
 * @param[in]  ctrl_flags  Flags controlling output formatting, see ::CTRL_FLAGS.
 * @param[in]  p_ind       Optional indents used for output formatting, may be @a NULL.
 */
void print_bit_mask_list( FILE *fp, const char *info_1, const char *info_2,
                          uint32_t supp_mask, int n_known, const char * const names[],
                          STR_FNC *s_fnc, int inst_idx, CTRL_FLAGS ctrl_flags,
                          const INDENTS *p_ind )
{
  const char *str_from_s_fnc = s_fnc ? s_fnc( inst_idx ) : mbgcfg_str_empty;

  #if DEBUG_MBGCFG_UTIL > 1
    fprintf( fp, "-- Begin bit mask list, " );

    if ( p_ind )
      fprintf( fp, "indent: %i, %i", p_ind->indent_2, p_ind->indent_3 );
    else
      fprintf( fp, "no indents" );

    fprintf( fp, "\n" );
  #endif

  if ( p_ind )
    print_indent( fp, p_ind->indent_2 );

  if ( ctrl_flags & CTRL_PRINT_ALL )
  {
    supp_mask = ( 1UL << n_known ) - 1;

    if ( info_1 )
    {
      if ( supp_mask )
      {
        // FIXME TODO Print a trailing space?
        fprintf( fp, "Known %s%s:", info_1, str_from_s_fnc );
        // FIXME TODO fprintf( fp, "%s%s:", info_1, str_from_s_fnc );
      }
      else
        fprintf( fp, "No %s%s known.", info_1, str_from_s_fnc );
    }
  }
  else
  {
    if ( info_1 )
    {
      if ( supp_mask )
      {
        fprintf( fp, "%s", info_1 );

        if ( info_2 )
          fprintf( fp, " %s", info_2 );

        #if DEBUG_MBGCFG_UTIL > 1
          fprintf( fp, " (%04lX)", (ulong) supp_mask );
        #endif

        fprintf( fp, ":" );
      }
      else
      {
        fprintf( fp, "No %s.", info_1 );

        if ( info_2 )
          fprintf( fp, " %s", info_2 );

        fprintf( fp, "." );
      }
    }
  }


  if ( supp_mask )
  {
    int n_printed = 0;
    int i;
    const char *str_sep = ( ctrl_flags & CTRL_PRINT_PLUS ) ? "+" : ", ";

    for ( i = 0; i < n_known; i++ )
    {
      const char *cp;

      if ( ( supp_mask & ( 1UL << i ) ) == 0 )
        continue;

      if ( names )
        cp = names[i];
      else
        if ( s_fnc )
          cp = s_fnc( i );
        else
          cp = mbgcfg_str_empty;

      if ( ctrl_flags & ( CTRL_PRINT_NEWLINES | CTRL_PRINT_IDX ) )
      {
        fprintf( fp, "\n" );

        if ( p_ind )
          print_indent( fp, p_ind->indent_3 );

        if ( ctrl_flags & CTRL_PRINT_IDX )
          fprintf( fp, "%i: ", i );

        fprintf( fp, "%s", cp );
      }
      else
        fprintf( fp, "%s%s", n_printed ? str_sep : mbgcfg_str_empty, cp );

      n_printed++;
    }
  }

  if ( !( ctrl_flags & CTRL_OMIT_FINAL_NEWLINE ) )
    fprintf( fp, "\n" );

  #if DEBUG_MBGCFG_UTIL > 1
    fprintf( fp, "-- End bit mask list\n" );
  #endif

}  // print_bit_mask_list



/*HDR*/
/**
 * @brief Check if parameter string starts with a particular keyword.
 *
 * @param[in]  s        The parameter string to check.
 * @param[in]  keyword  The keyword to be looked for.
 *
 * @return Pointer to the first character @a after the keyword in @p s,
 *         if the exact @p keyword was found in @p s, else @a NULL.
 */
const char *str_parm_p( const char *s, const char *keyword )
{
  const char *cp = NULL;
  const char *match = strstr( s, keyword );

  if ( match )
  {
    // The keyword was found as substring in s, but
    // other criteria need to be checked.

    size_t l;

    // The keyword must be at the beginning of s.
    if ( match != s )
      goto out;

    l = strlen( keyword );

    // The substring we found at the beginning of the
    // parameter string s must not be longer than the
    // keyword we are currently searching for.
    // If the character in s *after* the keyword is
    // an alpha character or numeric digit, the keyword
    // in the parameter string has the wrong length.
    if ( isalnum( s[l] ) )
      goto out;

    // We have found the keyword in the parameter string
    // and will return a pointer to the character *after*
    // the keyword in the parameter string.
    cp = &s[l];
  }

out:
  #if DEBUG_OPTIONS
    fprintf( MBGCFG_HELP_FP, "str_parm_p \"%s\" \"%s\": %s\n", s, keyword, cp ? "FOUND" : "" );
  #endif

  return cp;

}  // str_parm_p



/*HDR*/
void usage_newline( void )
{
  fprintf( MBGCFG_HELP_FP, "%s", "\n" );

}  // usage_newline



/*HDR*/
__attribute__( ( format( printf, 4, 5 ) ) )
int usage_line( const INDENTS *p_ind, const MBGCFG_OPT_INFO *p_oi,
                const char *cmd_parm, const char *cmd_comment_fmt, ... )
{
  FILE *fp = MBGCFG_HELP_FP;
  char s[256];
  size_t max_len = sizeof( s ) - 1;
  int n = 0;

  // Print left margin, if not 0.
  if ( p_ind->left_marg )
    n += snprintf_safe( &s[n], max_len - n, "%*s", p_ind->left_marg, mbgcfg_str_empty );

  // Print command name.
  if ( p_oi )
    n += snprintf_safe( &s[n], max_len - n, "%s", p_oi->cmd_name );

  // Print the command parameters, if specified.
  if ( cmd_parm )
  {
    if ( p_oi && ( p_oi->opt_flags & OPT_FLAG_CMD_IDX ) )
      n += snprintf_safe( &s[n], max_len - n, "%s", "<n>" );

    n += snprintf_safe( &s[n], max_len - n, "=%s", cmd_parm );
  }

  // Print the command comment, which can be a format string
  // expecting additional parameters.
  if ( cmd_comment_fmt )
  {
    va_list arg_list;

    // Indent the comment string.
    if ( p_ind->indent_2 )
    {
      int w = p_ind->indent_2 - n;

      // If the current string already ends after the tab position,
      // add at least 2 spaces before the comment.
      if ( w < 1 )  // TODO or < 2 ?
        w = 2;

      n += snprintf_safe( &s[n], max_len - n, "%*s", w, mbgcfg_str_empty );
    }

    va_start( arg_list, cmd_comment_fmt );
    n += vsnprintf_safe( &s[n], max_len - n, cmd_comment_fmt, arg_list );
    va_end( arg_list );
  }

  fprintf( fp, "%s\n", s );

  return n;

}  // usage_line



/*HDR*/
int print_indent( FILE *fp, int i )
{
  int n = 0;

  if ( i )
    n += fprintf( fp, "%*s", i, mbgcfg_str_empty );

  return n;

}  // print_indent



/*HDR*/
void usage_line_where( const INDENTS *p_ind )
{
  FILE *fp = MBGCFG_HELP_FP;

  if ( p_ind )
    print_indent( fp, p_ind->indent_note );

  fprintf( fp, "%s\n", "where:" );

}  // usage_line_where



/*HDR*/
__attribute__( ( format( printf, 2, 3 ) ) )
int usage_note( int indent, const char *fmt, ... )
{
  FILE *fp = MBGCFG_HELP_FP;

  // Print left margin, if not 0.
  int n = print_indent( fp, indent );

  if ( fmt )
  {
    va_list arg_list;

    va_start( arg_list, fmt );
    n += vfprintf( fp, fmt, arg_list );
    n += fprintf( fp, "\n" );
    va_end( arg_list );
  }

  return n;

}  // usage_note



