
/**************************************************************************
 *
 *  $Id: mbg_prog_info.c 1.1 2025/07/10 14:50:03Z martin.burnicki REL_M $
 *
 *  Functions to create and print common program info strings.
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 * -----------------------------------------------------------------------
 *  $Log: mbg_prog_info.c $
 *  Revision 1.1  2025/07/10 14:50:03Z  martin.burnicki
 *  Initial revision with code from toolutil.c/.h.
 *
 **************************************************************************/

#define _MBG_PROG_INFO
  #include <mbg_prog_info.h>
#undef _MBG_PROG_INFO

// Include Meinberg headers.
#include <str_util.h>

// Include system headers.
#include <stdio.h>

#if defined( MBG_TGT_WIN32 )
  #include <io.h>   // for _isatty()
#endif


#if defined( MBG_TGT_WIN32 ) && defined( _MSC_VER )
  // Required for MSC, but not for e.g. MinGW.
  // Must be defined *after* stdio has been included.
  #define isatty   _isatty
  #define fileno   _fileno
#endif



/*HDR*/
/**
 * @brief Make stdout unbuffered.
 *
 * If the output has been redirected, make stdout unbuffered, e.g.
 * to see the output immediately even though piped through 'tee'.
 */
void mbg_make_stdout_unbuffered( void )
{
  if ( !isatty( fileno( stdout ) ) )
    setvbuf( stdout, NULL, _IONBF, 0 );

}  // mbg_make_stdout_unbuffered



/*HDR*/
/**
 * @brief Print the program version to a string buffer.
 *
 * @param[out]  s        Address of a string buffer to be filled.
 * @param[in]   max_len  Size of the string buffer.
 *
 * @return The number of characters written to the string buffer,
 *         except the terminating 0.
 */
int mbg_program_version_str( char *s, size_t max_len )
{
  int n = snprintf_safe( s, max_len, "%s", MBG_FULL_VERSION_STR );

  return n;

}  // mbg_program_version_str



/*HDR*/
/**
 * @brief Print some program info to a string buffer.
 *
 * @param[out]  s           Address of a string buffer to be filled.
 * @param[in]   max_len     Size of the string buffer.
 * @param[in]   pname       The program name.
 * @param[in]   first_year  First copyright year.
 * @param[in]   last_year   Last copyright year.
 *
 * @return The number of characters written to the string buffer,
 *         except the terminating 0.
 */
int mbg_program_info_str( char *s, size_t max_len, const char *pname,
                          int first_year, int last_year )
{
  int n = 0;

  if ( last_year == 0 )
    last_year = MBG_CURRENT_COPYRIGHT_YEAR;

  n += snprintf_safe( &s[n], max_len - n, "%s v", pname );

  n += mbg_program_version_str( &s[n], max_len - n );

  n += snprintf_safe( &s[n], max_len - n, " copyright Meinberg " );

  if ( first_year != last_year )
    n += snprintf_safe( &s[n], max_len - n, "%04i-", first_year );

  n += snprintf_safe( &s[n], max_len - n, "%04i", last_year );

  return n;

}  // mbg_program_info_str



/*HDR*/
/**
 * @brief Print program info to console.
 *
 * @param[in]  pname       The program name.
 * @param[in]  first_year  First copyright year.
 * @param[in]  last_year   Last copyright year.
 */
void mbg_print_program_info( const char *pname, int first_year, int last_year )
{
  char ws[256];

  // If the output has been redirected, make stdout unbuffered, e.g.
  // to see the output immediately even though piped through 'tee'.
  mbg_make_stdout_unbuffered();

  mbg_program_info_str( ws, sizeof( ws ), pname, first_year, last_year );

  printf( "\n%s\n\n", ws );

}  // mbg_print_program_info


