
/**************************************************************************
 *
 *  $Id: mbgclock_main.c 1.6 2025/11/17 11:53:10Z martin.burnicki REL_M $
 *
 *  Description:
 *    Main file for for mbgclock driver to support Meinberg bus level
 *    devices under FreeBSD.
 *
 *    The binary is a loadable module called mbgclock which implements
 *    /dev/mbgclock* devices.
 *
 *    Based on FreeBSD's mypci.c sample program by Murray Stokely.
 *
 * -----------------------------------------------------------------------
 *  $Log: mbgclock_main.c $
 *  Revision 1.6  2025/11/17 11:53:10Z  martin.burnicki
 *  Cleanup and improved debug builds.
 *  Revision 1.5  2025/06/04 16:26:36Z  martin.burnicki
 *  Consider changes made in pcpsdrvr.c
 *  Revision 1.4  2025/06/04 13:02:30Z  martin.burnicki
 *  Consider API changes in newer FreeBSD kernels, where the
 *  DRIVER_MODULE macro no longer expects the 'devclass' argument.
 *  Revision 1.3.1.5  2023/03/08 11:03:18Z  martin.burnicki
 *  Account for changes in mbgversion.h.
 *  Revision 1.3.1.4  2020/10/15 10:56:33  martin
 *  Account for modified logging from kernel space.
 *  Revision 1.3.1.3  2018/06/13 16:03:10  martin
 *  Account for updated log feature.
 *  Revision 1.3.1.2  2018/01/24 16:53:49  martin
 *  Fixed build.
 *  Revision 1.3.1.1  2017/10/18 11:02:21  martin
 *  *** empty log message ***
 *  Revision 1.3  2017/08/10 14:43:15  martin
 *  String 'pcps_driver_name' is now defined in pcpsdrvr.h.
 *  Account for unified extended resource properties handling.
 *  Revision 1.2  2017/07/06 09:24:03  martin
 *  Implemented IOCTL handler which basically can also
 *  supports privilege checking for IOCTL.
 *  Improved resource handling.
 *  Use 'struct thread' instead of obsolete 'd_thread_t' type.
 *  Reported by John Baldwin and George Neville-Neil (FreeBSD bug #196692).
 *  Unified handling of program version information.
 *  Support DEBUG messages.
 *  Code cleanup.
 *  Revision 1.1  2011/01/26 13:56:32  martin
 *  Initial skeleton based on FreeBSD's mypci.c sample program by Murray Stokely.
 *
 **************************************************************************/

#include <pcpsdrvr.h>
#include <mbgddmsg.h>
#include <mbgversion.h>
#include <mbgioctl.h>

#include <sys/param.h>        /* defines used in kernel.h */
#include <sys/module.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <sys/kernel.h>       /* types used in module initialization */
#include <sys/conf.h>         /* cdevsw struct */
#include <sys/uio.h>          /* uio struct */
#include <sys/malloc.h>
#include <sys/bus.h>          /* structs, prototypes for pci bus stuff */

#include <sys/rman.h>

#include <dev/pci/pcivar.h>   /* For pci_get macros! */
#include <dev/pci/pcireg.h>



#define MBG_MICRO_VERSION_CODE   0

#define MBG_COPYRIGHT    "(c) Meinberg 2010-" MBG_CURRENT_COPYRIGHT_YEAR_STR


#if USE_DEBUG_PORT
  static MBG_DBG_DATA mbg_dbg_data;
  static MBG_DBG_PORT mbg_dbg_port = 0x378 + 0;  //##++
#endif  // defined( DEBUG )

static PCPS_DRVR_INFO drvr_info = { MBG_SHORT_VERSION_CODE,
                                    0, MBG_DRVR_NAME " v" MBG_NUMERIC_VERSION_STR };

MALLOC_DEFINE( M_MBGCLOCK, "short desc", "long desc" );

#include <macioctl.h>



/* The softc holds our per-instance data. */
struct mbgclock_softc
{
  device_t device;
  struct cdev *cdev;
  PCPS_DDEV *pddev;
};


/* Function prototypes */
static d_open_t mbgclock_open;
static d_close_t mbgclock_close;
static d_read_t mbgclock_read;
static d_write_t mbgclock_write;
static d_ioctl_t mbgclock_ioctl;


/* Character device entry points */

static struct cdevsw mbgclock_cdevsw =
{
  .d_version = D_VERSION,
  .d_open = mbgclock_open,
  .d_close = mbgclock_close,
  .d_read = mbgclock_read,
  .d_write = mbgclock_write,
  .d_ioctl = mbgclock_ioctl,
  .d_name = "mbgclock"
};



static __mbg_inline
void set_dev_connected( PCPS_DDEV *pddev, int state )
{
  _mbgddmsg_3( DEBUG_DRVR, MBG_LOG_INFO, "Setting dev %s_%s connected state to %i",
               _pcps_ddev_type_name( pddev ), _pcps_ddev_sernum( pddev ), state );
  atomic_store_rel_int( &pddev->connected, state );

}  // set_dev_connected



/*
 * In the cdevsw routines, we find our softc by using the si_drv1 member
 * of struct cdev.  We set this variable to point to our softc in our
 * attach routine when we create the /dev entry.
 */
int
mbgclock_open( struct cdev *dev, int oflags, int devtype, struct thread *td )
{
  struct mbgclock_softc *psc = dev->si_drv1;
  PCPS_DDEV *pddev = psc->pddev;

  _mbgddmsg_2( DEBUG_DRVR, MBG_LOG_INFO, "Device %s_%s opened successfully.",
               _pcps_ddev_type_name( pddev ), _pcps_ddev_sernum( pddev ) );

  atomic_add_int( &pddev->open_count, 1 );

  return 0;

}  // mbgclock_open



int
mbgclock_close( struct cdev *dev, int fflag, int devtype, struct thread *td )
{
  struct mbgclock_softc *psc = dev->si_drv1;
  PCPS_DDEV *pddev = psc->pddev;

  atomic_subtract_int( &pddev->open_count, 1 );

  _mbgddmsg_2( DEBUG_DRVR, MBG_LOG_INFO, "Device %s_%s closed.",
               _pcps_ddev_type_name( pddev ), _pcps_ddev_sernum( pddev ) );

  return 0;

}  // mbgclock_close



int
mbgclock_read( struct cdev *dev, struct uio *uio, int ioflag )
{
  #if defined( DEBUG )
    struct mbgclock_softc *psc = dev->si_drv1;
    PCPS_DDEV *pddev = psc->pddev;

    _mbgddmsg_3( DEBUG_DRVR, MBG_LOG_INFO, "Device %s_%s asked to read %li bytes",
                 _pcps_ddev_type_name( pddev ), _pcps_ddev_sernum( pddev ),
                 (long) uio->uio_resid );
  #endif

  return 0;

}  // mbgclock_read



int
mbgclock_write( struct cdev *dev, struct uio *uio, int ioflag )
{
  #if defined( DEBUG )
    struct mbgclock_softc *psc = dev->si_drv1;
    PCPS_DDEV *pddev = psc->pddev;

    _mbgddmsg_3( DEBUG_DRVR, MBG_LOG_INFO, "Device %s_%s asked to write %li bytes",
                 _pcps_ddev_type_name( pddev ), _pcps_ddev_sernum( pddev ),
                 (long) uio->uio_resid );
  #endif

  return 0;

}  // mbgclock_write



int
mbgclock_ioctl( struct cdev *dev, u_long cmd, caddr_t data,
                int32_t flag, struct thread *td )  // credentials in thread
{
  struct mbgclock_softc *psc = dev->si_drv1;
  PCPS_DDEV *pddev = psc->pddev;
  int priv_lvl;

  // Find out which privilege level is required
  // to execute this IOCTL command.
  priv_lvl = ioctl_get_required_privilege( cmd );

  // Check if the calling process has the required privilege.
  switch ( priv_lvl )
  {
    case MBG_REQ_PRIVL_NONE:
      // Always allow.
      break;

    case MBG_REQ_PRIVL_EXT_STATUS:
    case MBG_REQ_PRIVL_CFG_READ:
      // This may require some privilege for the calling process.
      // Anyway, always allow for now.
      break;

    case MBG_REQ_PRIVL_CFG_WRITE:
    case MBG_REQ_PRIVL_SYSTEM:
      #if 0    // TODO Check if required privileges are available, e.g.:
        if ( !capable( CAP_SYS_ADMIN ) )  // don't allow if no root privileges
        {
          _mbgddmsg_4( DEBUG_DRVR, MBG_LOG_INFO, "%p IOCTL 0x%02lX: permission denied, dev %s_%s",
                       dev, cmd, _pcps_ddev_type_name( pddev ), _pcps_ddev_sernum( pddev ) );
          return EPERM;
        }
        //##+++++  !capable(CAP_SYS_ADMIN) is also used in FreeBSD's xfs_ioctl.c
      #endif
      break;

    default:
      _mbgddmsg_4( DEBUG_DRVR, MBG_LOG_WARN, "%p IOCTL 0x%02lX: unknown, permission denied, dev %s_%s",
                   dev, cmd, _pcps_ddev_type_name( pddev ), _pcps_ddev_sernum( pddev ) );
      return IOCTL_RC_ERR_PERM;
  }

  // If the return value is not 0 then this is considered as an error code
  // which is stored in errno, and lets the ioctl call return -1, so the
  // calling user space code has to read errno if -1 is returned.
  // TODO check if the statement above is correct.

  return ioctl_switch( pddev, cmd, (void *) data, (void *) data );

}  // mbgclock_ioctl



/* PCI Support Functions */

static void
mbg_deallocate_resource( device_t dev, EXT_RSRC_INFO *p_ri, int type )
{
  if ( p_ri->res )
  {
    bus_deactivate_resource( dev, type, p_ri->rid, p_ri->res );
    bus_release_resource( dev, type, p_ri->rid, p_ri->res );
    p_ri->res = NULL;
  }

}  // mbg_deallocate_resource



/*
 * deallocate resources
 */
static void
mbg_dealloc_rsrcs( device_t dev )
{
  struct mbgclock_softc *psc = device_get_softc( dev );
  PCPS_DDEV *pddev = psc->pddev;
  PCPS_RSRC_INFO *prsrci = &pddev->rsrc_info;
  int i;

  // mbg_deallocate_resource( dev, &prsrci->irq.bsd, SYS_RES_IRQ );

  for ( i = 0; i < N_PCPS_MEM_RSRC; i++ )
    mbg_deallocate_resource( dev, &prsrci->mem[i].ext, SYS_RES_MEMORY );

  for ( i = 0; i < N_PCPS_PORT_RSRC; i++ )
    mbg_deallocate_resource( dev, &prsrci->port[i].ext, SYS_RES_IOPORT );

}  // mbg_dealloc_rsrcs



static void
mbg_alloc_rsrc( device_t dev, int rid, EXT_RSRC_INFO *p_ri, int type, int flags )
{
  p_ri->rid = rid;

  p_ri->res = bus_alloc_resource_any( dev, type, &p_ri->rid, flags );

  if ( p_ri->res )
  {
    p_ri->bst = rman_get_bustag( p_ri->res );
    p_ri->bsh = rman_get_bushandle( p_ri->res );
  }

}  // mbg_alloc_rsrc



static void
mbg_alloc_rsrcs( device_t dev )
{
  struct mbgclock_softc *psc = device_get_softc( dev );
  PCPS_DDEV *pddev = psc->pddev;
  PCPS_RSRC_INFO *prsrci = &pddev->rsrc_info;
  EXT_RSRC_INFO ri;
  int bar;

  #if DEBUG_RSRC
    mbg_kdd_msg( MBG_LOG_INFO, "Allocating rsrcs for PCI device 0x%04X:0x%04X",
                 pci_get_vendor( dev ), pci_get_device( dev ) );
  #endif

  for ( bar = 0; bar < 5; bar ++ )
  {
    int rid = PCIR_BAR( bar );

    if ( prsrci->num_rsrc_io < N_PCPS_PORT_RSRC )
    {
      mbg_alloc_rsrc( dev, rid, &ri, SYS_RES_IOPORT, RF_ACTIVE );

      if ( ri.res )
      {
        prsrci->port[prsrci->num_rsrc_io].ext = ri;
        pcps_add_rsrc_io( pddev, rman_get_start( ri.res ), rman_get_size( ri.res ) );
        continue;
      }
    }

    if ( prsrci->num_rsrc_mem < N_PCPS_MEM_RSRC )
    {
      mbg_alloc_rsrc( dev, rid, &ri, SYS_RES_MEMORY, RF_ACTIVE );

      if ( ri.res )
      {
        prsrci->mem[prsrci->num_rsrc_mem].ext = ri;
        pcps_add_rsrc_mem( pddev, rman_get_start( ri.res ), rman_get_size( ri.res ) );
        continue;
      }
    }
  }


  // single IRQ resource
#if 0  // currently not used / required
  mbg_alloc_rsrc( dev, 0, &ri, SYS_RES_IRQ, RF_SHAREABLE | RF_ACTIVE );

  if ( ri.res )
  {
    prsrci->port[prsrci->num_rsrc_irq].bsd = ri;
    pcps_add_rsrc_irq( pddev, rman_get_start( ri.res ) );
  }
#endif

}  // mbg_alloc_rsrcs



/*
 * Probe: compare the device ID of this device against the IDs that this driver
 * supports.  If there is a match, set the description and return success.
 */
static int
mbgclock_probe( device_t dev )
{
  _mbgddmsg_fnc_entry();

  uint16_t vend_id = pci_get_vendor( dev );
  uint16_t dev_id = pci_get_device( dev );
  PCPS_DEV_TYPE_EX *pdt;


  if ( vend_id != PCI_VENDOR_MEINBERG )
    goto fail;

  pdt = pcps_get_dev_type_table_entry( PCPS_BUS_PCI, dev_id );

  if ( pdt == NULL )
    goto fail;

  device_set_desc( dev, pdt->dev_type.name );

  _mbgddmsg_2( DEBUG_DEV_INIT, MBG_LOG_INFO, "probe: PCI device 0x%04X:0x%04X supported.",
               vend_id, dev_id );

  return BUS_PROBE_DEFAULT;


fail:
  _mbgddmsg_2( DEBUG_DEV_INIT, MBG_LOG_INFO, "probe: PCI device 0x%04X:0x%04X not supported",
               vend_id, dev_id );
  return ENXIO;

}  // mbgclock_probe



/*
 * Attach function is only called if the probe is successful.
 */
static int
mbgclock_attach( device_t dev )
{
  uint16_t dev_id = pci_get_device( dev );
  struct mbgclock_softc *psc = device_get_softc( dev );
  int rc;
  int sys_rc;

  _mbgddmsg_1( DEBUG_DEV_INIT, MBG_LOG_INFO, "Attach for device 0x%04X",
               dev_id );

  rc = pcps_init_ddev( &psc->pddev );

  if ( mbg_rc_is_error( rc ) )
  {
    _mbgddmsg_2( DEBUG_DEV_INIT, MBG_LOG_INFO, "Attach device 0x%04X: pcps_init_ddev() failed, rc: %i",
                 dev_id, rc );
    sys_rc = ENOMEM;
    goto fail;
  }


  rc = pcps_setup_ddev( psc->pddev, PCPS_BUS_PCI, dev_id );

  if ( mbg_rc_is_error( rc ) )
  {
    _mbgddmsg_2( DEBUG_DEV_INIT, MBG_LOG_INFO, "Attach device 0x%04X: pcps_setup_ddev() failed, rc: %i",
                 dev_id, rc );
    sys_rc = ENXIO;
    goto fail;
  }


  //##++++++ TODO rc = pci_enable_device( dev );


  mbg_alloc_rsrcs( dev );

  //##++++++  TODO rc = pcps_probe_device( pddev, device->bus->number, device->devfn );
  rc = pcps_probe_device( psc->pddev, 0, 0 );

  if ( mbg_rc_is_error( rc ) )
  {
    _mbgddmsg_2( DEBUG_DEV_INIT, MBG_LOG_INFO, "Attach device 0x%04X: pcps_probe_device() failed, rc: %i",
                 dev_id, rc );
    sys_rc = ENXIO;
    goto fail;
  }


  set_dev_connected( psc->pddev, 1 );

  /* Initialize our softc. */
  psc->device = dev;
  /*
   * Create a /dev entry for this device.  The kernel will assign us
   * a major number automatically.  We use the unit number of this
   * device as the minor number and name the character device
   * "mbgclock<unit>".
   */
  psc->cdev = make_dev( &mbgclock_cdevsw, device_get_unit( dev ),
    UID_ROOT, GID_WHEEL, 0600, "mbgclock%u", device_get_unit( dev ) );

  psc->cdev->si_drv1 = psc;

  _mbgddmsg_1( DEBUG_DEV_INIT, MBG_LOG_INFO, "Device 0x%04X attached successfully.",
               dev_id );

  #if TEST_MM_ACCESS_TIME
    if ( psc->pddev->mm_ts.valid )
      test_mm_access_time( psc->pddev );
  #endif

  return 0;


fail:
  mbg_dealloc_rsrcs( dev );

  if ( psc->pddev )
    pcps_cleanup_ddev( psc->pddev );

  _mbgddmsg_1( DEBUG_DEV_INIT, MBG_LOG_INFO, "Failed to attach device 0x%04X.",
               dev_id );
  return sys_rc;

}  // mbgclock_attach



/*
 * Detach device.
 */
static int
mbgclock_detach( device_t dev )
{
  #if defined( DEBUG )
    uint16_t dev_id = pci_get_device( dev );
  #endif
  struct mbgclock_softc *psc;
  PCPS_DDEV *pddev;

  _mbgddmsg_1( DEBUG_DEV_INIT, MBG_LOG_INFO, "Detach for device 0x%04X",
               dev_id );

  /* Teardown the state in our softc created in our attach routine. */
  psc = device_get_softc( dev );
  pddev = psc->pddev;

  set_dev_connected( pddev, 0 );

//  pcps_free_ddev( pddev );  //##++++++  should wait for outstanding requests
  mbg_dealloc_rsrcs( dev );

  if ( psc->pddev )
    pcps_cleanup_ddev( psc->pddev );

  psc->pddev = NULL;

  destroy_dev( psc->cdev );

  _mbgddmsg_1( DEBUG_DEV_INIT, MBG_LOG_INFO, "Device 0x%04X detached",
               dev_id );

  return 0;

}  // mbgclock_detach



/*
 * Called during system shutdown after sync.
 */
static int
mbgclock_shutdown( device_t dev )
{
  printf( "mbgclock shutdown!\n" );

  return 0;

}  // mbgclock_shutdown



/*
 * Device suspend routine.
 */
static int
mbgclock_suspend( device_t dev )
{
  printf( "mbgclock suspend!\n" );

  return 0;

}  // mbgclock_suspend



/*
 * Device resume routine.
 */
static int
mbgclock_resume( device_t dev )
{
  printf( "mbgclock resume!\n" );

  return 0;

}  // mbgclock_resume



static device_method_t mbgclock_methods[] =
{
  /* Device interface */
  DEVMETHOD( device_probe,     mbgclock_probe ),
  DEVMETHOD( device_attach,    mbgclock_attach ),
  DEVMETHOD( device_detach,    mbgclock_detach ),
  DEVMETHOD( device_shutdown,  mbgclock_shutdown ),
  DEVMETHOD( device_suspend,   mbgclock_suspend ),
  DEVMETHOD( device_resume,    mbgclock_resume ),
  { 0, 0 }
};



DEFINE_CLASS_0( mbgclock, mbgclock_driver, mbgclock_methods, sizeof( struct mbgclock_softc ) );

// Since commit 9e4e7bcaef3fcaab66cdecd08b1664ac92cc9a6b to the FreeBSD kernel source,
// the DRIVER_MODULE macro no longer expects the 'devclass' argument.
// This change first appeared in release/14.0.0~7814, and the first __FreeBSD_version
// that contains the change is 1400067.

#if __FreeBSD_version >= 1400067
DRIVER_MODULE( mbgclock, pci, mbgclock_driver, 0, 0 );
#else
static devclass_t mbgclock_devclass;
DRIVER_MODULE( mbgclock, pci, mbgclock_driver, mbgclock_devclass, 0, 0 );
#endif
