/**************************************************************************
 *
 *  $Id: meinberg.c $
 *  $Name: $
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Example program to access radio clocks via LabWindows instrument driver
 *    to the kernel device driver.
 *
 *  Build environment settings:
 *    The meinberg driver need to be installed in version later then 2.09. 
 *
 * -----------------------------------------------------------------------
 *  $Log: meinberg.c $
 *
 **************************************************************************/
#include "meinberg.h" 	    
/**************************************************************************/
char *sprint_hr_time( char *s, const PCPS_HR_TIME *t )
{
  sprintf( s, "%08lX.%08lX UTC%+ldsec, st: %04X", 
           t->tstamp.sec,
           t->tstamp.frac,
           t->utc_offs,
           t->status
         );

  return s;

}  // sprint_hr_time
/**************************************************************************/
int print_date_time( PCPS_TIME *tp, char* buffer)
{
  int rc;
  
  rc = sprintf(buffer,  "%02u.%02u.%02u  %02u:%02u:%02u.%02u (UTC%+dh)\n",
          			     tp->mday, tp->month, tp->year, 
          			     tp->hour, tp->min, tp->sec, tp->sec100,
          			     tp->offs_utc );
  
  //Error handling: success = zero
  if(rc > 0)
     rc = 0;          			
    
  return rc;          			
} // print_date_time  
/**************************************************************************/
int get_ref_offs_from_string(const char* p, int base)
{
  int std,min,offs;

  sscanf(p, "%d:%d", &std, &min);
  offs = (abs(std) * MINS_PER_HOUR + abs(min)) * MINS_PER_HOUR;

  if (base == MINS)
    offs /= MINS_PER_HOUR;

  if (std < 0)
    offs = -offs;

  return offs;

}  // get_ref_offs_from_string
/**************************************************************************/
char *str_time_short_from_offs( char *s,int offs, int base )
{
  int h, min;

  if (base == MINS)
    offs *= MINS_PER_HOUR;

  h = offs / SECS_PER_HOUR;
  min = ( offs % SECS_PER_HOUR ) / MINS_PER_HOUR;

  sprintf( s, "%2.2d:%2.2d",
           h, min );

  return s;

}  // str_time_short_from_offs
/**************************************************************************/
int mbg_print_device_name( MBG_DEV_HANDLE dh, PCPS_DEV *p_dev, int with_port_base, char* device_name)
{
  int rc;
  PCPS_ASIC_VERSION asic_version;
  
  memset(device_name, 0, sizeof(device_name));

  rc = mbg_get_device_info( dh, p_dev );

  if ( rc == PCPS_SUCCESS )
  {
    if(with_port_base)
       rc = sprintf(device_name, "%s at port %03Xh", _pcps_type_name( p_dev ), _pcps_port_base( p_dev, 0 ));
    else
       rc = sprintf(device_name, "%s", _pcps_type_name( p_dev ));
       
    //Error handling: success = zero
    if(rc > 0)
       rc = 0;
  }
  else
  {
    sprintf(device_name, "Failed to read device info");
    return -1;
  }
  
  //read pci asic version if available and append it to the device name string
  if ( _pcps_is_pci_asic( p_dev ) )
      if ( mbg_get_asic_version( dh, &asic_version ) == PCPS_SUCCESS )
         sprintf( &device_name[strlen( device_name )], " (PCI ASIC v%lu)", asic_version );
    
  return rc;
}  // mbg_print_device_name
/**************************************************************************/
int mbg_print_driver_info(MBG_DEV_HANDLE dh, char* driver_info)
{
  int rc;
  PCPS_DRVR_INFO drvr_info;
  
  if ( dh == MBG_INVALID_DEV_HANDLE )
  {
     strcpy(driver_info, "Unable to open device");
     return -1;
  }

  rc = mbg_get_drvr_info( dh, &drvr_info );

  if ( rc != PCPS_SUCCESS )
  {
    strcpy(driver_info, "Failed to read driver info");
    return -2;
  }
  
  rc = sprintf( driver_info, "Kernel driver: %s v%i.%02i\n\n",
                drvr_info.id_str, drvr_info.ver_num / 100,
                drvr_info.ver_num % 100);
                
  //Error handling: success = zero
  if(rc > 0)
     rc = 0;
     
  return rc;          
}  // mbg_print_driver_info
/**************************************************************************/
int mbg_print_gps_pos( MBG_DEV_HANDLE dh, char* gpspos)
{
  POS pos;
  int rc;
  
  rc = sizeof(pos);
  rc = sizeof(pos.lla);
  rc = sizeof(pos.xyz);
  rc = sizeof(pos.longitude.sec);
  

  if ( PCPS_SUCCESS == mbg_get_gps_pos( dh, &pos ))
  {
    sprint_pos_geo( gpspos, &pos, ", ", 4 );
  }
  else
  {
    sprintf(gpspos, "Failed to read receiver position");
    return -1;
  }
    
  return 0;  
}  // mbg_print_gps_pos
/**************************************************************************/
int mbg_print_gps_sv_info( MBG_DEV_HANDLE dh, char* sv_info)
{
  int rc;
  STAT_INFO stat_info;
  
  if ( PCPS_SUCCESS == mbg_get_gps_stat_info( dh, &stat_info ) )
  {
    rc = sprintf(sv_info, "Satellites: %u in view, %u good\n",
               stat_info.svs_in_view, 
               stat_info.good_svs );
               
	//Error handling: success = zero
    if(rc > 0)
       rc = 0;               
  }
  else
  {
    sprintf(sv_info, "Failed to read GPS satellite info");
    return -1;
  }
    
  return 0;  
}  // mbg_print_sv_info
/**************************************************************************/
int mbg_print_status( MBG_DEV_HANDLE dh, char* status, char* status2)
{
  PCPS_TIME t;

  if ( PCPS_SUCCESS == mbg_get_time( dh, &t ) )
  {
     sprintf(status, "%s\n", ( t.status & PCPS_FREER ) ? "free running" : "synchronized" );

     sprintf(status2, "%s\n", ( t.status & PCPS_SYNCD ) ? "synchronized after last RESET" : "not synchronized after last RESET" );
  }
  else
  {
     sprintf(status, "Failed to read status");
     return -1;
  }
    
  return 0;   
}  // mbg_print_status
/**************************************************************************/
int mbg_print_sync_time(MBG_DEV_HANDLE dh, char* sync_time)
{
   int rc;
   PCPS_TIME tp;

   rc = mbg_get_sync_time(dh, &tp);
   
   if(rc == PCPS_SUCCESS)
      rc = print_date_time(&tp, sync_time);
   else 
      return -1;
   
   return rc;
} // mbg_print_sync_time
/**************************************************************************/
int mbg_print_time(MBG_DEV_HANDLE dh, char* time)
{
   int rc;
   PCPS_TIME tp;
   
   rc = mbg_get_time(dh, &tp);
   
   if(rc == PCPS_SUCCESS)
      rc = print_date_time(&tp, time);
   else
      return -1;
      
   return rc;
}  // mbg_print_time
/**************************************************************************/
int mbg_print_antenna_field (MBG_DEV_HANDLE dh, char* field)
{
   int rc;
   PCPS_TIME tp;
   
   rc = mbg_get_time(dh, &tp);
   
   if(rc == PCPS_SUCCESS)
      sprintf(field, "%d", tp.signal);
   else
      return -1;
      
   return rc;
}
/**************************************************************************/
int mbg_print_antenna_status(MBG_DEV_HANDLE dh, char* ant_status)
{
   int rc;
   int val;
   PCPS_TIME tp;
   
   rc = mbg_get_time(dh, &tp);
   
   if(rc == PCPS_SUCCESS)
   {
      val = tp.signal - PCPS_SIG_BIAS;

      if(val <= PCPS_SIG_ERR)
         sprintf(ant_status, "antenna faulty");
      else
         sprintf(ant_status, "Normal operation");
   }
   else
   {
      sprintf(ant_status, "failed to read antenna status");
      return -1;
   }

   return 0;
}
/**************************************************************************/
int mbg_print_gps_ant_cable_len(MBG_DEV_HANDLE dh, char* cable)
{
   int rc;
   ANT_CABLE_LEN ant_cab_len;		// Antenna cable length structure
   
   rc = mbg_get_gps_ant_cable_len (dh, &ant_cab_len);
   
   if(rc == PCPS_SUCCESS)
   {
      sprintf(cable, "%um", ant_cab_len);
   }
   else
   {
   	   sprintf(cable, "failed to read cable length");
   	   return -1;
   }
   
   return 0;
   
}
/**************************************************************************/
int mbg_print_irig_signal(MBG_DEV_HANDLE dh, char* signal)
{
   int rc;
   PCPS_TIME tp;
   
   rc = mbg_get_time(dh, &tp);
   
   if(rc == PCPS_SUCCESS)
      sprintf(signal, "%d", tp.signal);
   else
      return -1;
      
   return rc;
}
/**************************************************************************/
int mbg_print_irig_status(MBG_DEV_HANDLE dh, char* irig_status)
{
   int rc;
   int val;
   PCPS_TIME tp;
   
   rc = mbg_get_time(dh, &tp);
   
   if(rc == PCPS_SUCCESS)
   {
      val = tp.signal - PCPS_SIG_BIAS;

      if(val <= PCPS_SIG_ERR)
         sprintf(irig_status, "no input signal");
      else
         sprintf(irig_status, "Normal operation");
   }
   else
   {
      sprintf(irig_status, "failed to read IRIG status");
      return -1;
   }

   return 0;
}
/**************************************************************************/
int mbg_print_hr_time( MBG_DEV_HANDLE dh, char* hrtime)
{
  PCPS_HR_TIME hr_t;

  if ( PCPS_SUCCESS == mbg_get_hr_time( dh, &hr_t ) )
    sprint_hr_time( hrtime, &hr_t );
  else
    sprintf(hrtime, "Failed to read High Resolution Time" );
    
  return 0;  

}  // mbg_print_hr_time
/**************************************************************************/
int mbg_print_time_cycles(MBG_DEV_HANDLE dh, PCPS_DEV *dev, char* timecycles)
{   
   PCPS_TIME_CYCLES tc;
   PCPS_HR_TIME_CYCLES hr_tc;   
   
   if(_pcps_has_hr_time( dev ))   
   {
      mbg_get_hr_time_cycles(dh, &hr_tc);
      sprintf(timecycles, "%I64u", hr_tc.cycles.QuadPart);
   }
   else
   {
      mbg_get_time_cycles(dh, &tc);
      sprintf(timecycles, "%u %u", tc.cycles);
   }
   
   return 0;   
} //mbg_print_time_cycles
/**************************************************************************/
