/**************************************************************************
 *
 *  $Id: main.c $
 *  $Name: $
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Example program to access radio clocks via LabWindows instrument driver
 *    to the kernel device driver.
 *
 *  Build environment settings:
 *    The meinberg driver need to be installed in version later then 2.09.
 *
 *  Author: Gregoire Diehl (gregoire.diehl@meinberg.de)
 *
 * -----------------------------------------------------------------------
 *  $Log: main.c $
 *
 **************************************************************************/
#include <meinberg.h> 

#include <mbgdevio.h>
#include <gpsutils.h>

#include "main_panel.h"

#include <time.h> 

#include <userint.h>
#include <utility.h>

/**************************************************************************/
int main_panel; 				// handle to user interface file  

int device_index;       	    // index of the selected device

char stemp[200];       			// string format buffer

int devices_found;   			// number of radio clocks found in the system
char device_name[50];  			// device name buffer
char driver_info[50];  			// driver info buffer

static PCPS_DEV dev; 			// device structure
MBG_DEV_HANDLE dh;   			// device handle structure: dh is a logical unique identifier to a device

RECEIVER_INFO rec_info; 		// receiver info structure
RECEIVER_PORT_CFG rec_port_cfg; // serial settings structure

ENABLE_FLAGS enable_flags;      // GPS enable flags structure

IRIG_INFO irig_info;		    // IRIG information structure

PCPS_REF_OFFS ref_offs;			// REF_OFFS information structure

char *Baud[]  = MBG_BAUD_STRS;           // Array of available baud rates
char *Frame[] = MBG_FRAMING_STRS;		 // Array of available framings
char *Mode[]  = DEFAULT_ENG_MODE_NAMES;  // Array of available modes

/**************************************************************************/
int CVICALLBACK SetAntennaCableLength (int panel, int control, int event, void *callbackData, int eventData1, int eventData2)
{
	if(event == EVENT_COMMIT)
	{
	   ANT_CABLE_LEN ant_cable_len;
	   
	   //Get new cable length
	   GetCtrlVal(main_panel, PANEL_STRING_STAT_SET, stemp);
	   
	   ant_cable_len = atoi(stemp);
	   
	   //if negative value: set new length to 0
	   if(ant_cable_len < 0)
	      ant_cable_len = 0;
	   
	   //Sending new cable length to device
	   mbg_set_gps_ant_cable_len (dh, &ant_cable_len);
	   
	   // No error handling here: if the cmd was successfully called: the new cable length
	   // was refreshed in the cable length field
	   
	   //Remove new cable length from component
	   SetCtrlVal(main_panel, PANEL_STRING_STAT_SET, "");
	}
	
	return 0;
}
/**************************************************************************/
int CVICALLBACK ClearUcapEvent (int panel, int control, int event, void *callbackData, int eventData1, int eventData2)
{
	if(event == EVENT_COMMIT)
	{
	   int rc; 
	   
	   //Clear all internal stored user capture of the device
	   rc = mbg_clr_ucap_buff (dh);
	   
	   //On success: Flash green LED for one second
	   if(rc == PCPS_SUCCESS)
	   {
	      SetCtrlVal(main_panel, PANEL_LED_UCAP, 1);
          ProcessDrawEvents();
          
          Delay(1);
       
          SetCtrlVal(main_panel, PANEL_LED_UCAP, 0);
	   }
	   else  //On failure: Flash red LED for one second
	   {
	   	  //change color
          SetCtrlAttribute (main_panel, PANEL_LED_UCAP, ATTR_ON_COLOR, VAL_RED);
          SetCtrlVal(main_panel, PANEL_LED_UCAP, 1);
          ProcessDrawEvents();
           
          Delay(1);
        
          SetCtrlVal(main_panel, PANEL_LED_UCAP, 0);
          SetCtrlAttribute (main_panel, PANEL_LED_UCAP, ATTR_ON_COLOR, VAL_GREEN);
          
          return -1;
	   }
	}
	
	return 0;
}
/**************************************************************************/
int CVICALLBACK GetUcapEvent (int panel, int control, int event, void *callbackData, int eventData1, int eventData2)
{
	if(event == EVENT_COMMIT)
	{
	   int rc;
	   PCPS_HR_TIME hr_time;
	   PCPS_UCAP_ENTRIES ucap_entries;
	   
	   struct tm *gmt_t;
	   time_t tt;
	   
	   //Get number of available user capture events of the device
	   rc = mbg_get_ucap_entries (dh, &ucap_entries);
	   
	   if(rc == PCPS_SUCCESS)
	   {
	      //if capture event available
	      if(ucap_entries.used > 0)
	      {
	         //Get next user capture event from device as HR time structure with enhanced accuracy
	         rc = mbg_get_ucap_event (dh, &hr_time);
	   
	         //On success: display time and fraction else error_msg
	         if(rc == PCPS_SUCCESS)
	         {
				tt = hr_time.tstamp.sec;
				gmt_t = gmtime (&tt);
				
	            sprintf(stemp, "%.2d.%.2d.%.2d %.2d:%.2d:%.2d.%.2u", gmt_t->tm_mday, gmt_t->tm_mon, gmt_t->tm_year+1970, gmt_t->tm_hour, gmt_t->tm_min, gmt_t->tm_sec, hr_time.tstamp.frac);
	            SetCtrlVal(main_panel, PANEL_STRING_UCAP_EVENT, stemp);
	         }
	         else
	         {
	            SetCtrlVal(main_panel, PANEL_STRING_UCAP_EVENT, "failed to read capture event");
	            return -1;
	         }
   	      }   
   	      else
   	         SetCtrlVal(main_panel, PANEL_STRING_UCAP_EVENT, "no capture event available");
	   }
	   else
	   {
	      SetCtrlVal(main_panel, PANEL_STRING_UCAP_EVENT, "failed");
	      return -2;
	   }
	}
	
	return 0;
}
/**************************************************************************/
int DisplayUCapEvents(MBG_DEV_HANDLE dh, PCPS_DEV* dev)
{
   int rc;
   TTM ttm;
   char cap_event[50];
   PCPS_UCAP_ENTRIES ucap_entries;
   
   memset(cap_event, 0, sizeof(cap_event));
   
   //check if device supports new capture event functions
   if(_pcps_has_ucap (dev))
   {
	  //Get number of available user capture events
	  rc = mbg_get_ucap_entries (dh, &ucap_entries);
	  
	  //Display used/max value on success else error_msg
	  if(rc == PCPS_SUCCESS)
      {
         sprintf(stemp, "%u/%u", ucap_entries.used, ucap_entries.max);
         SetCtrlVal(main_panel, PANEL_STRING_UCAP, stemp);
      }
      else
      {
   	     sprintf(stemp, "failed");
   	     SetCtrlVal(main_panel, PANEL_STRING_UCAP, stemp);
   	     return -1;
      }
   }
   else
   {
      //no ucap support for available number of events / clear buffer: disable section partial
      SetCtrlAttribute (main_panel, PANEL_STRING_UCAP, ATTR_DIMMED, 1);
      SetCtrlAttribute (main_panel, PANEL_COMMANDBUTTON_UCAP_CL, ATTR_DIMMED, 1);
      SetCtrlAttribute (main_panel, PANEL_COMMANDBUTTON_UCAP_GE, ATTR_DIMMED, 1);
      
      //check if capture event available
	  rc = mbg_get_gps_ucap (dh, &ttm);
	  
	  if(rc == PCPS_SUCCESS)
	  {
	     //if structure is not filled: no capture
	     if(ttm.tm.year != 0)
	     {
	        sprintf(cap_event, "%.2d.%.2d.%.2d %.2d:%.2d:%.2d.%.2d",ttm.tm.mday, ttm.tm.month, ttm.tm.year, ttm.tm.hour, ttm.tm.min, ttm.tm.sec, ttm.tm.frac);
	        
	        //Display event
	        SetCtrlVal(main_panel, PANEL_STRING_UCAP_EVENT, cap_event);
	     }
	  }
	  else
	  {  //Display error
	     SetCtrlVal(main_panel, PANEL_STRING_UCAP_EVENT, "failed to read ucap events");
	     return -2;
	  }
   }
   
   return 0;
}
/**************************************************************************/
int CVICALLBACK SaveIrigSettings (int panel, int control, int event, void *callbackData, int eventData1, int eventData2)
{
	if(event == EVENT_COMMIT)
	{
	   int ic;
	   int rc;
	   int rc2;
	   
	   //Get new irig settings from IRIG section components
	   GetCtrlVal(main_panel, PANEL_RING_IRIG, &ic);
	   GetCtrlVal(main_panel, PANEL_STRING_IRIG, stemp);
	   
	   //Set new IRIG-code in strucutre irig_info
	   irig_info.settings.icode = ic;
	   
	   //Convert the inserted IRIG-Offset string to an integer
	   ref_offs = get_ref_offs_from_string(stemp, MINS);
	   
	   //Save new IRIG-settings 
	   rc = mbg_set_irig_rx_settings (dh, &irig_info.settings);
	   
	   //Save new IRIG-Offset
	   rc2 = mbg_set_ref_offs (dh, &ref_offs);  
	   
	   //If both successful: Flash green LED for one second
	   if((rc | rc2) == PCPS_SUCCESS)
	   {
	      SetCtrlVal(main_panel, PANEL_LED_IRIG, 1);
          ProcessDrawEvents();           
          
          Delay(1);
       
          SetCtrlVal(main_panel, PANEL_LED_IRIG, 0);
	   }
	   else //On failure: Flash red LED for one second
	   {
	   	  //change color
          SetCtrlAttribute (main_panel, PANEL_LED_IRIG, ATTR_ON_COLOR, VAL_RED);
          SetCtrlVal(main_panel, PANEL_LED_IRIG, 1);
          ProcessDrawEvents ();           
          
          Delay(1);
        
          SetCtrlVal(main_panel, PANEL_LED_IRIG, 0);
          SetCtrlAttribute (main_panel, PANEL_LED_IRIG, ATTR_ON_COLOR, VAL_GREEN);
          
          return -1;
	   }
 	}
	return 0;
}
/**************************************************************************/
int GetIrigInfo(MBG_DEV_HANDLE dh)
{
   int i;
   int rc;
   int index;
   ulong s;	
   char *icode[] = DEFAULT_ICODE_RX_NAMES;
   
   //Receive irig_info structure from device
   rc = mbg_get_irig_rx_info (dh, &irig_info);
   
   if(rc == PCPS_SUCCESS)
   {
      //Delete all IRIG-Code entries of ring component
      ClearListCtrl (main_panel, PANEL_RING_IRIG);   
      
      //s is a bit-mask representing the available IRIG-codes
      s = irig_info.supp_codes;
      
      //Inserted all suported IRIG-Codes to ring coponent
      for ( i = 0; i < N_ICODE_RX; i++ )
      {
         if ( s & 1 )
            InsertListItem (main_panel, PANEL_RING_IRIG, -1, icode[i], i);   
           
         s>>=1;
      }
      
      //Set the actual selected IRIG-code of the device as default
      GetIndexFromValue (main_panel, PANEL_RING_IRIG, &index, irig_info.settings.icode);
 	  SetCtrlAttribute (main_panel, PANEL_RING_IRIG, ATTR_CTRL_INDEX, index);
   }
   else
      return -1;
      
   //Receive ref_offset value from device
   rc = mbg_get_ref_offs (dh, &ref_offs);
   
   if(rc == PCPS_SUCCESS)
   {
      //if value is out of range: display out of range msg
      if ( _pcps_ref_offs_out_of_range( ref_offs ))
         SetCtrlVal (main_panel, PANEL_STRING_IRIG, "o/r");
      else
      {
         //format ref_offset to a hh:mm value and write it to component
         str_time_short_from_offs( stemp, ( ref_offs ), 0 );
         SetCtrlVal (main_panel, PANEL_STRING_IRIG, stemp);
      }
   }
   else
   {
      SetCtrlVal (main_panel, PANEL_STRING_IRIG, "error");
      return -2;
   }
   
   return 0;
}

/**************************************************************************/
int CVICALLBACK SaveDeviceFlags (int panel, int control, int event, void *callbackData, int eventData1, int eventData2)
{
	if(event == EVENT_COMMIT)
	{
	   int rc;
	   int pu;
	   int se;
	   int sy;
	   
       //Get new enable flags values: 0 - disable;1  - enable
       GetCtrlVal(main_panel, PANEL_RING_PULSES, &pu);
       GetCtrlVal(main_panel, PANEL_RING_SERIAL, &se);
       GetCtrlVal(main_panel, PANEL_RING_SYNTH, &sy);
       
       //if set: enable pulses
       if(pu)
          enable_flags.pulses = EF_PULSES_BOTH;
       else
          enable_flags.pulses = EF_OFF;
       
       //if set: enable serial port
       if(se)
          enable_flags.serial = EF_SERIAL_BOTH;
       else
          enable_flags.serial = EF_OFF;
          
       //if set: enable fsynth
       if(sy)
          enable_flags.synth = EF_SYNTH;
       else
          enable_flags.synth = EF_OFF;
   
       //Send new settings to the device
       rc = mbg_set_gps_enable_flags (dh, &enable_flags);
   
       //if the call was succesful: Flash green LED for one second
       if(rc == PCPS_SUCCESS)
       {
          SetCtrlVal(main_panel, PANEL_LED_FLAGS, 1);
          ProcessDrawEvents ();           
          
          Delay(1);
       
          SetCtrlVal(main_panel, PANEL_LED_FLAGS, 0);
       } 
       else //On failure: Flash red LED for one second
       {
          //change color
          SetCtrlAttribute (main_panel, PANEL_LED_FLAGS, ATTR_ON_COLOR, VAL_RED);
          SetCtrlVal(main_panel, PANEL_LED_FLAGS, 1);
          ProcessDrawEvents ();           
          
          Delay(1);
        
          SetCtrlVal(main_panel, PANEL_LED_FLAGS, 0);
          SetCtrlAttribute (main_panel, PANEL_LED_FLAGS, ATTR_ON_COLOR, VAL_GREEN);
          
          return -1;
       }
	}
	
	return 0;
}
/**************************************************************************/
int GetDeviceFlags(MBG_DEV_HANDLE dh)
{
	 int rc;
	 
	 //Get enable flags settings from the device: enable_flags values are bool values
	 rc = mbg_get_gps_enable_flags (dh, &enable_flags);
	 
	 //Setting the components to "if sync" or "always"
	 if(rc==PCPS_SUCCESS)
	 {
	    SetCtrlVal(main_panel, PANEL_RING_PULSES, (int)enable_flags.pulses);
	    SetCtrlVal(main_panel, PANEL_RING_SERIAL, (int)enable_flags.serial);
	    SetCtrlVal(main_panel, PANEL_RING_SYNTH, (int)enable_flags.synth);
	 }
	 else //failure: no 
	    return -1;
	    
	 return 0;   
}
/**************************************************************************/ 
void GetSerialSettings(MBG_DEV_HANDLE dh, PCPS_DEV* dev)
{
   int index;
   int rc=0;
   int cnt;
   ulong s;									 
   int i;
   int com_num;
   char temp[50];
   char temp2[50];
   
   memset(temp2, 0, sizeof(temp2));
   memset(&rec_info, 0, sizeof(rec_info));
   memset(&rec_port_cfg, 0, sizeof(rec_port_cfg));
   
   //Setup receiver info structure: receiver_info contains information about the device: like number of available com ports 
   rc = mbg_setup_receiver_info (dh, dev, &rec_info);
   
   if(rc == PCPS_SUCCESS)
   {
      //Get serial setting from device
      rc = mbg_get_serial_settings (dh, dev, &rec_port_cfg, &rec_info);
   
      if(rc == PCPS_SUCCESS)
      {
		 //get selected comport
		 GetCtrlAttribute (main_panel, PANEL_NUMERIC_COMPORT, ATTR_CTRL_VAL, &com_num);
         
         //port_cfg has zero based index
         com_num--;
         
         //Insert available baudrate for this device & com port to the ring component
         ClearListCtrl (main_panel, PANEL_RING_BAUD);   
         s = rec_port_cfg.pii[com_num].port_info.supp_baud_rates;
         for ( i = 0; i < N_MBG_BAUD_RATES; i++ )
         {
            if ( s & 1 )
               InsertListItem (main_panel, PANEL_RING_BAUD, -1, Baud[i], i);   
            
            s>>=1;
         }
         
		 //Find the selected baudrate of the device in the ring component and select it
		 GetNumListItems (main_panel, PANEL_RING_BAUD, &cnt);
         for(i=0;i<cnt;i++)
         {
			GetLabelFromIndex (main_panel, PANEL_RING_BAUD, i, temp);
			
			sprintf(temp2, "%d", rec_port_cfg.pii[com_num].port_info.port_settings.parm.baud_rate);
			if(strcmp(temp, temp2) == 0)
			{
			   SetCtrlAttribute (main_panel, PANEL_RING_BAUD, ATTR_CTRL_VAL, i);
			   break;
			}
         }
         
         //Insert available framings for this device & com port to the ring component
         ClearListCtrl (main_panel, PANEL_RING_FRAME);   
         s = rec_port_cfg.pii[com_num].port_info.supp_framings;
         for ( i = 0; i < N_MBG_FRAMINGS; i++ )
         {
            if ( s & 1 )
               InsertListItem (main_panel, PANEL_RING_FRAME, -1, Frame[i], i);   
            
            s>>=1;
         }
         
         //Find the selected framing of the device in the ring component and select it
         GetNumListItems (main_panel, PANEL_RING_FRAME, &cnt);
         for(i=0;i<cnt;i++)
         {
			GetLabelFromIndex (main_panel, PANEL_RING_FRAME, i, temp);
			
			sprintf(temp2, "%s", rec_port_cfg.pii[com_num].port_info.port_settings.parm.framing);
			if(strcmp(temp, temp2) == 0)
			{
			   SetCtrlIndex (main_panel, PANEL_RING_FRAME, i);
			   break;
			}
         }
         
         //Insert available string types for this device & com port to the ring component
         ClearListCtrl (main_panel, PANEL_RING_STRING);   
         s = rec_port_cfg.pii[com_num].port_info.supp_str_types;
         for ( i = 0; i < MAX_PARM_STR_TYPE; i++ )
         {
            if ( s & 1 ) 
               InsertListItem (main_panel, PANEL_RING_STRING, -1, rec_port_cfg.stii[i].str_type_info.long_name, i);   
               
            s>>=1;
  		 }
  		 
  		 //Set the selected string type off the device as default
  		 GetIndexFromValue (main_panel, PANEL_RING_STRING, &index, rec_port_cfg.pii[com_num].port_info.port_settings.str_type);
		 SetCtrlAttribute (main_panel, PANEL_RING_STRING, ATTR_CTRL_INDEX, rec_port_cfg.pii[com_num].port_info.port_settings.str_type);
  		 
  		 //Insert available string modes for this device & com port & string type to the ring component
  		 ClearListCtrl (main_panel, PANEL_RING_MODE);
  		 cnt = rec_port_cfg.pii[com_num].port_info.port_settings.str_type;
  		 for ( i = 0; i < N_STR_MODE; i++ )
  		 {
            if (rec_port_cfg.stii[cnt].str_type_info.supp_modes & ( 1UL << i ) )
               InsertListItem (main_panel, PANEL_RING_MODE, -1, Mode[i], i);   
         }
         
		 //Set the selected string mode off the device as default
		 GetIndexFromValue (main_panel, PANEL_RING_MODE, &index, rec_port_cfg.pii[com_num].port_info.port_settings.mode);
		 SetCtrlAttribute (main_panel, PANEL_RING_MODE, ATTR_CTRL_INDEX, index);
      }
   }
}
/**************************************************************************/ 
int CVICALLBACK SaveSerialSettingsCallback (int panel, int control, int event, void *callbackData, int eventData1, int eventData2)
{
	int rc;
	int index;
	int com_num;
	char label[50];
	
	if(event == EVENT_COMMIT)
	{
	  //get selected comport
	  GetCtrlAttribute (main_panel, PANEL_NUMERIC_COMPORT, ATTR_CTRL_VAL, &com_num);
	  
	  //port_cfg has zero based index
      com_num--;
      
      //store baud
      GetCtrlIndex(main_panel, PANEL_RING_BAUD, &index);
	  GetLabelFromIndex (main_panel, PANEL_RING_BAUD, index, label);
      rec_port_cfg.pii[com_num].port_info.port_settings.parm.baud_rate = atoi(label);
      
      //store frame
      GetCtrlIndex(main_panel, PANEL_RING_FRAME, &index);
      GetLabelFromIndex (main_panel, PANEL_RING_FRAME, index, label);
      strcpy(rec_port_cfg.pii[com_num].port_info.port_settings.parm.framing, label);
      
      //store str type
      GetCtrlVal(main_panel, PANEL_RING_MODE, &index);
      rec_port_cfg.pii[com_num].port_info.port_settings.mode = index;
      
      //store str mode
      GetCtrlVal(main_panel, PANEL_RING_STRING, &index);
      rec_port_cfg.pii[com_num].port_info.port_settings.str_type = index;
      
      //Send new serial settings to the device
      rc = mbg_save_serial_settings (dh, &dev, &rec_port_cfg, com_num);
   
      //if the call was succesful: Flash green LED for one second
      if(rc == PCPS_SUCCESS)
      {
         SetCtrlVal(main_panel, PANEL_LED, 1);
    	 ProcessDrawEvents ();           
    	 
         Delay(1);
         
         SetCtrlVal(main_panel, PANEL_LED, 0);
      }
      else //On failure: Flash red LED for one second
      {
         //change color
		 SetCtrlAttribute (main_panel, PANEL_LED, ATTR_ON_COLOR, VAL_RED);
         SetCtrlVal(main_panel, PANEL_LED, 1);
         ProcessDrawEvents ();           
         
         Delay(1);
         
         SetCtrlVal(main_panel, PANEL_LED, 0);
         SetCtrlAttribute (main_panel, PANEL_LED, ATTR_ON_COLOR, VAL_GREEN);
         
         return -1;
      }
      
	}
	
	return 0;
}
/**************************************************************************/
int CVICALLBACK ChangeSerialStringTypeCallback (int panel, int control, int event, void *callbackData, int eventData1, int eventData2)
{
	if(event == EVENT_COMMIT)
	{
	   int i;
	   int new_str_type;
	   
	   //Get new selected str_type
	   GetCtrlVal(main_panel, PANEL_RING_STRING, &new_str_type);
	   
	   //Delete all inserted str_modes from ring component
  	   ClearListCtrl (main_panel, PANEL_RING_MODE);
  	   
  	   //Insert new str_modes for the corresponding str_type to the ring component
  	   for ( i = 0; i < N_STR_MODE; i++ )
  	   {
          if (rec_port_cfg.stii[new_str_type].str_type_info.supp_modes & ( 1UL << i ) )
             InsertListItem (main_panel, PANEL_RING_MODE, -1, Mode[i], i);   
       }
	}
	return 0;
}
/**************************************************************************/
int CVICALLBACK ChangeComportCallback (int panel, int control, int event, void *callbackData, int eventData1, int eventData2)
{
	if(event == EVENT_COMMIT)
	{
	   //if the comports of a device are toggled: call GetSerialSettings again to set the value 
	   //for the new selected comport
	   GetSerialSettings(dh, &dev);
	}
	return 0;
}
/**************************************************************************/
int loop (int panel, int control, int event, void *callbackData, int eventData1,  int eventData2)
{
  char buffer[MBG_STR_SIZE];
  char buffer2[MBG_STR_SIZE];
  
  if (event == EVENT_IDLE)
  {
     //if devices available
     if(devices_found > 0)
     {
        //Get device name
        if ( dh != MBG_INVALID_DEV_HANDLE )
        {
           //print time/date information
           mbg_print_time(dh, buffer);
           SetCtrlVal(main_panel, PANEL_STRING_TIME, buffer);
           
           // Some newer clocks (currently GPS only) support 
           // a high resolution time. If supported, read and 
           // display the HR time.
           if(_pcps_has_hr_time( &dev )) 
           {
              mbg_print_hr_time(dh, buffer);
              SetCtrlVal(main_panel, PANEL_STRING_HR_TIME, buffer);
           }
           else
              SetCtrlVal(main_panel, PANEL_STRING_HR_TIME, "no high resolution time supported");
           
           //print and display last synchronisation of the device
           mbg_print_sync_time(dh, buffer);
           SetCtrlVal(main_panel, PANEL_STRING_LAST_SYNC, buffer);
           
           //print and display status of the device
           mbg_print_status(dh, buffer, buffer2);
           SetCtrlVal(main_panel, PANEL_STRING_STATUS, buffer);
           SetCtrlVal(main_panel, PANEL_STRING_STATUS_2, buffer2);
           
           //get device info structure from device handle
           mbg_get_device_info( dh, &dev );
           
           //if the device is GPS based
           if (_pcps_is_gps( &dev )) 
           {
              //print sv information & gps position & antenna cable length
              mbg_print_gps_sv_info(dh, buffer);
              SetCtrlVal(main_panel, PANEL_STRING_STAT1, buffer);
              
              mbg_print_gps_pos(dh, buffer);
              SetCtrlVal(main_panel, PANEL_STRING_STAT2, buffer);
              
              mbg_print_gps_ant_cable_len (dh, buffer);
              SetCtrlVal(main_panel, PANEL_STRING_STAT3, buffer);
              
              DisplayUCapEvents(dh, &dev);
           }
           
           //if the device is DCF based
           if(_pcps_is_dcf( &dev ))
           {
              //print antenna status & field strength
              mbg_print_antenna_status(dh, buffer);
              mbg_print_antenna_field (dh, buffer2);
              
              //display values
              SetCtrlVal(main_panel, PANEL_STRING_STAT1, buffer);
              SetCtrlVal(main_panel, PANEL_STRING_STAT2, buffer2);
           }
           
           //if the device is IRIG based
           if(_pcps_is_irig_rx ( &dev ))
           {
			  //get irig status & signal 
			  mbg_print_irig_status(dh, buffer);
			  mbg_print_irig_signal(dh, buffer2);
			  
			  //display values
			  SetCtrlVal(main_panel, PANEL_STRING_STAT1, buffer);
              SetCtrlVal(main_panel, PANEL_STRING_STAT2, buffer2);
           }
           
           //get performance counter & display value
           mbg_print_time_cycles(dh, &dev, buffer);
           SetCtrlVal(main_panel, PANEL_STRING_PERF_COUNTER, buffer);
        }
     }
     
     ProcessSystemEvents();
  }
  return(0);
}
/**************************************************************************/
int CVICALLBACK QuitCallback (int panel, int control, int event,void *callbackData, int eventData1, int eventData2)
{
	if(event == EVENT_COMMIT)
	{
	   QuitUserInterface(0); 
	}
	
	return 0;
}
/**************************************************************************/
int CVICALLBACK change_device_callback (int panel, int control, int event, void *callbackData, int eventData1, int eventData2)
{
	if(event == EVENT_COMMIT)
	{
	   int rc;
	   
	   //Clear handle
       mbg_close_device(&dh);   
       
	   //get selected device from ring component
	   GetCtrlVal(main_panel, PANEL_RING_DEVICES, &device_index);
	   
	   //open device to get a valid device handle
       dh = mbg_open_device( device_index ); 
       
       //Read device info structure from current device to identify the clock (model name, serial number, firmware rev.)
       rc = mbg_get_device_info( dh, &dev );   
       
       if(rc == PCPS_SUCCESS)
       {
          //get driver info   
          rc = mbg_print_driver_info (dh, driver_info);
        
          if(rc == PCPS_SUCCESS)
              SetCtrlVal(main_panel, PANEL_STRING_DRIVER_INFO, driver_info);
           else
              SetCtrlVal(main_panel, PANEL_STRING_DRIVER_INFO, "n/a");
          
          //display serial number of the selected device
          if(strlen(_pcps_sernum( &dev )) > 0)
             SetCtrlVal(main_panel, PANEL_STRING_SERIAL, _pcps_sernum( &dev ));    
          else
             SetCtrlVal(main_panel, PANEL_STRING_SERIAL, "not available");    
          
          //read and/or setup receiver info
          rc = mbg_setup_receiver_info (dh, &dev, &rec_info);
       
          //Insert number of available comports into ring component
          if(rc == PCPS_SUCCESS)
          {
             SetCtrlAttribute (main_panel, PANEL_NUMERIC_COMPORT, ATTR_MAX_VALUE, rec_info.n_com_ports);
             SetCtrlAttribute (main_panel, PANEL_NUMERIC_COMPORT, ATTR_DIMMED, 0);
             
             //if only one com port available: disable arrows else: enable arrows
             if(rec_info.n_com_ports == 1)
				SetCtrlAttribute (main_panel, PANEL_NUMERIC_COMPORT, ATTR_SHOW_INCDEC_ARROWS, 0);
             else   
                SetCtrlAttribute (main_panel, PANEL_NUMERIC_COMPORT, ATTR_SHOW_INCDEC_ARROWS, 1);
          }
          else //Error reading number of available com ports: disable control
	         SetCtrlAttribute (main_panel, PANEL_NUMERIC_COMPORT, ATTR_DIMMED, 1);
	 	      
	      //Set numeric control for the numbers of com ports to the first entry
	      SetCtrlAttribute (main_panel, PANEL_NUMERIC_COMPORT, ATTR_CTRL_VAL, 1);
          
          //If the current one is a GPS based device
          if (_pcps_is_gps( &dev ))  
          {
          	  //overwrite header 
			  SetCtrlAttribute (main_panel, PANEL_STRING_STAT1, ATTR_LABEL_TEXT, "SV info:");
			  SetCtrlAttribute (main_panel, PANEL_STRING_STAT2, ATTR_LABEL_TEXT, "Position:");
			  SetCtrlAttribute (main_panel, PANEL_STRING_STAT3, ATTR_LABEL_TEXT, "Cable len:");
			  
          	  //enable flags section
          	  SetCtrlAttribute (main_panel, PANEL_COMMANDBUTTON_SAVE_FL, ATTR_DIMMED, 0);
              SetCtrlAttribute (main_panel, PANEL_RING_PULSES, ATTR_DIMMED, 0);
              SetCtrlAttribute (main_panel, PANEL_RING_SERIAL, ATTR_DIMMED, 0);
              
              //enable ucap section
              SetCtrlAttribute (main_panel, PANEL_COMMANDBUTTON_UCAP_CL, ATTR_DIMMED, 0);
              SetCtrlAttribute (main_panel, PANEL_COMMANDBUTTON_UCAP_GE, ATTR_DIMMED, 0);
              SetCtrlAttribute (main_panel, PANEL_STRING_UCAP, ATTR_DIMMED, 0);
              SetCtrlAttribute (main_panel, PANEL_STRING_UCAP_EVENT, ATTR_DIMMED, 0);
              
              //if the device support synthesizer: enable ring, else disable ring
              if(rec_info.features & GPS_HAS_SYNTH)
                 SetCtrlAttribute (main_panel, PANEL_RING_SYNTH, ATTR_DIMMED, 0);
              else
                 SetCtrlAttribute (main_panel, PANEL_RING_SYNTH, ATTR_DIMMED, 1);
                 
              GetDeviceFlags(dh);   
              
              //Clear last capture event from component
              SetCtrlVal(main_panel, PANEL_STRING_UCAP_EVENT, "");
          }
          else
          {
              //disable flags section
              SetCtrlAttribute (main_panel, PANEL_COMMANDBUTTON_SAVE_FL, ATTR_DIMMED, 1);
              SetCtrlAttribute (main_panel, PANEL_RING_PULSES, ATTR_DIMMED, 1);
              SetCtrlAttribute (main_panel, PANEL_RING_SERIAL, ATTR_DIMMED, 1);
              SetCtrlAttribute (main_panel, PANEL_RING_SYNTH, ATTR_DIMMED, 1);
              
              //disable ucap section
              SetCtrlAttribute (main_panel, PANEL_COMMANDBUTTON_UCAP_CL, ATTR_DIMMED, 1);
              SetCtrlAttribute (main_panel, PANEL_COMMANDBUTTON_UCAP_GE, ATTR_DIMMED, 1);
              SetCtrlAttribute (main_panel, PANEL_STRING_UCAP, ATTR_DIMMED, 1);
              SetCtrlAttribute (main_panel, PANEL_STRING_UCAP_EVENT, ATTR_DIMMED, 1);
          }
          
          //If the current one is a DCF based device
          if (_pcps_is_dcf( &dev ))  
          {
              //overwrite header 
			  SetCtrlAttribute (main_panel, PANEL_STRING_STAT1, ATTR_LABEL_TEXT, "Antenna:");
			  SetCtrlAttribute (main_panel, PANEL_STRING_STAT2, ATTR_LABEL_TEXT, "Field:");
			  SetCtrlAttribute (main_panel, PANEL_STRING_STAT3, ATTR_LABEL_TEXT, "");
          }
          
          //If the current one is a IRIG based receiver 
          if(_pcps_is_irig_rx ( &dev ))
          {
             //overwrite header 
			 SetCtrlAttribute (main_panel, PANEL_STRING_STAT1, ATTR_LABEL_TEXT, "IRIG signal:");
			 SetCtrlAttribute (main_panel, PANEL_STRING_STAT2, ATTR_LABEL_TEXT, "Value:");
			 
             //enable irig section
          	 SetCtrlAttribute (main_panel, PANEL_COMMANDBUTTON_SAVE_IR, ATTR_DIMMED, 0);
             SetCtrlAttribute (main_panel, PANEL_RING_IRIG, ATTR_DIMMED, 0);
             SetCtrlAttribute (main_panel, PANEL_STRING_IRIG, ATTR_DIMMED, 0);
             
             //Get and display the current IRIG settings from the device
             GetIrigInfo(dh);
          }
          else
          {
             //disable irig section
          	 SetCtrlAttribute (main_panel, PANEL_COMMANDBUTTON_SAVE_IR, ATTR_DIMMED, 1);
             SetCtrlAttribute (main_panel, PANEL_RING_IRIG, ATTR_DIMMED, 1);
             SetCtrlAttribute (main_panel, PANEL_STRING_IRIG, ATTR_DIMMED, 1);
          }
          
          if(_pcps_has_cab_len( &dev))
          {
             //display antenna cable length section
		     SetCtrlAttribute (main_panel, PANEL_STRING_STAT_SET, ATTR_VISIBLE, 1);
		     SetCtrlAttribute (main_panel, PANEL_COMMANDBUTTON_SET_ANT, ATTR_VISIBLE, 1);
		     
			 SetCtrlAttribute (main_panel, PANEL_STRING_STAT3, ATTR_WIDTH, 160);
			 SetCtrlAttribute (main_panel, PANEL_STRING_STAT3, ATTR_VISIBLE, 1);
		  }
		  else
		  {
		     //hide antenna cable length section
		     SetCtrlAttribute (main_panel, PANEL_STRING_STAT_SET, ATTR_VISIBLE, 0);
		     SetCtrlAttribute (main_panel, PANEL_COMMANDBUTTON_SET_ANT, ATTR_VISIBLE, 0);
		     
		     SetCtrlAttribute (main_panel, PANEL_STRING_STAT3, ATTR_WIDTH, 230);
		     SetCtrlAttribute (main_panel, PANEL_STRING_STAT3, ATTR_VISIBLE, 0);
		
		  }
		  
		  //align label text right justified to the control
		  SetCtrlAttribute (main_panel, PANEL_STRING_STAT1, ATTR_LABEL_JUSTIFY,  VAL_RIGHT_JUSTIFIED);
		  SetCtrlAttribute (main_panel, PANEL_STRING_STAT2, ATTR_LABEL_JUSTIFY,  VAL_RIGHT_JUSTIFIED);
		  SetCtrlAttribute (main_panel, PANEL_STRING_STAT3, ATTR_LABEL_JUSTIFY,  VAL_RIGHT_JUSTIFIED);
		  SetCtrlAttribute (main_panel, PANEL_NUMERIC_COMPORT, ATTR_LABEL_JUSTIFY,  VAL_RIGHT_JUSTIFIED);
          
          //Get an display the serial settings (baud rate, framing, str mode, str type) of the current device
          GetSerialSettings(dh, &dev);
       }
       else
          return -1;
       
	   //Set the QUIT button as active control
	   SetActiveCtrl (main_panel, PANEL_COMMANDBUTTON);
    }
    
	return 0;
}
/**************************************************************************/
int main(void)
{
  int i;
  int rc;
  
  //Load uir file...
  main_panel = LoadPanel (0, "main_panel.uir", PANEL);
  
  //... and display it
  DisplayPanel(main_panel);

  //Install loop callback with intervall of 100ms
  InstallMainCallback (loop, 0, 1);  
  SetIdleEventRate (100);            
  
  //Version check: The version of the dll must match the version of the used API
  if (mbgdevio_check_version( MBGDEVIO_VERSION ) != PCPS_SUCCESS )
  {
     sprintf(stemp, "The MBGDEVIO DLL API version %X which is installed is not compatible\n"
                    "with API version %X required by this program.\n",
                     mbgdevio_get_version(), MBGDEVIO_VERSION);
            
     MessagePopup("Message", stemp);            
  }
  else
  {
     //get mbgdevio version and display it.
     rc = mbgdevio_get_version();
     sprintf(stemp, "%X", rc);
     SetCtrlVal(main_panel, PANEL_STRING_VERSION, stemp);
     
     //get number of present radio clocks
     devices_found = mbg_find_devices ();   
  
     //display number of available devices
     SetCtrlVal(main_panel, PANEL_NUMERIC_NUMBERDEVICES, devices_found);
  
     //insert all available clocks into ring component
     for(i=0;i<devices_found;i++)
     {
        //Open device to get a valid device handle
        dh = mbg_open_device( i );
        
        //Get device name
        if ( dh == MBG_INVALID_DEV_HANDLE )
           sprintf(stemp, "Unable to open device %d", i);
        else
           mbg_print_device_name(dh, &dev, 1, stemp);  
        
        //Insert into ring component
        InsertListItem (main_panel, PANEL_RING_DEVICES, i, stemp, i);   
        
        //Clear handle
        mbg_close_device(&dh);  
     }
  
     //If at least one device found: call change_device_callback to start with the first device
     if(devices_found > 0)
        change_device_callback (0, 0, EVENT_COMMIT, 0, 0, 0);
     
     //Start user interface
     RunUserInterface();         
  }

  return 0;
}
/**************************************************************************/
