
/**************************************************************************
 *
 *  $Id: pcpsdev.h 1.22 2004/11/09 12:57:52Z martin TEST $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Definitions used to share information on radio clock devices
 *    between device drivers which have direct access to the hardware
 *    devices and user space programs which evaluate and present that
 *    information.
 *
 *    At the bottom of the file there are some macros defined which
 *    should be used to access the structures to extract characteristics
 *    of an individual clock.
 *
 * -----------------------------------------------------------------------
 *  $Log: pcpsdev.h $
 *  Revision 1.22  2004/11/09 12:57:52Z  martin
 *  Redefined interface data types using C99 fixed-size definitions.
 *  Added support for TCR167PCI.
 *  New macro _pcps_has_gps_data().
 *  New type PCPS_STATUS_PORT.
 *  Removed obsolete inclusion of asm/timex.h for Linux.
 *  Revision 1.21  2004/09/06 15:19:49Z  martin
 *  Support a GPS_DATA interface where sizes are specified 
 *  by 16 instead of the original 8 bit quantities, thus allowing 
 *  to transfer data blocks which exceed 255 bytes.
 *  Modified inclusion of header files under Linux.
 *  Modified definition of MBG_PC_CYCLES for Linux.
 *  Revision 1.20  2004/04/14 09:09:11  martin
 *  Source code cleanup.
 *  Revision 1.19  2004/04/07 09:49:14Z  martin
 *  Support new feature PCPS_HAS_IRIG_TX.
 *  New macros _pcps_has_irig(), _pcps_has_irig_tx().
 *  Revision 1.18  2004/01/14 11:02:14Z  MARTIN
 *  Added formal type MBG_PC_CYCLES for OS/2,
 *  though it's not really required or used.
 *  Revision 1.17  2003/12/22 15:40:16  martin
 *  Support higher baud rates for TCR510PCI and PCI510.
 *  Supports PCPS_HR_TIME for TCR510PCI.
 *  New structures used to read device time together with associated
 *  PC CPU cycles.
 *  For Win32, differentiate between kernel mode and non-kernel mode.
 *  Moved some definitions here from mbgdevio.h.
 *  New type PCPS_ASIC_VERSION.
 *  New macro _pcps_ref_offs_out_of_range().
 *  Revision 1.16  2003/06/19 09:48:30Z  MARTIN
 *  Renamed symbols ..clr_cap_buffer to ..clr_ucap_buffer.
 *  New macro _pcps_has_ucap().
 *  New definitions to support cmds PCPS_GIVE_UCAP_ENTRIES
 *  and PCPS_GIVE_UCAP_EVENT.
 *  Revision 1.15  2003/04/15 09:57:25  martin
 *  New typedefs ALL_STR_TYPE_INFO, ALL_PORT_INFO,
 *  RECEIVER_PORT_CFG.
 *  Revision 1.14  2003/04/09 14:07:01Z  martin
 *  Supports PCI510, GPS169PCI, and TCR510PCI,
 *  and new PCI_ASIC used by those devices.
 *  Renamed macro _pcps_is_irig() to _pcps_is_irig_rx().
 *  New macros _pcps_has_ref_offs(), _pcps_has_opt_flags().
 *  Fixed macro _pcps_has_hr_time().
 *  New type PCPS_BUS_FLAGS.
 *  Preliminary support for PCPS_TZDL.
 *  Revision 1.13  2002/08/09 07:19:49  MARTIN
 *  Moved definition of ref time sources to pcpsdefs.h.
 *  New feature PCPS_CAN_CLR_CAP_BUFF and
 *  associated macro _pcps_can_clr_cap_buff().
 *  New macros _pcps_is_irig(), _pcps_has_signal(),
 *  _pcps_has_mod().
 *  Revision 1.12  2002/02/19 09:22:53  MARTIN
 *  Added definitions for the maximum number of clocks' serial ports
 *  and string types that can be handled by the configuration programs.
 *  Revision 1.11  2002/02/01 11:36:58  MARTIN
 *  Added new definitions for GPS168PCI.
 *  Inserted definitions of firmware REV_NUMs  for supported features
 *  which had previously been defined in pcpsdefs.h.
 *  Include use_pack.h.
 *  Updated comments.
 *  Source code cleanup.
 *  Revision 1.10  2001/11/30 09:52:48  martin
 *  Added support for event_time which, however, requires
 *  a custom GPS firmware.
 *  Revision 1.9  2001/10/16 10:11:14  MARTIN
 *  New Macro _pcps_has_serial_hs() which determines whether
 *  DCF77 clock supports baud rate higher than default.
 *  Re-arranged order of macro definitions.
 *  Revision 1.8  2001/09/03 07:15:05  MARTIN
 *  Added macro to access the firmware revision number.
 *  Cleaned up macro syntax.
 *  Added some comments.
 *  Revision 1.7  2001/08/30 13:20:04  MARTIN
 *  New macro to mark a PCPS_TIME variable  as unread.
 *  New macro to check if a PCPS_TIME variable  is unread.
 *  Revision 1.6  2001/03/15 15:45:01  MARTIN
 *  Added types PCPS_ERR_FLAGS, PCPS_BUS_NUM, PCPS_SLOT_NUM.
 *  Revision 1.5  2001/03/01 13:53:10  MARTIN
 *  Initial version for the new driver library.
 *
 **************************************************************************/

#ifndef _PCPSDEV_H
#define _PCPSDEV_H

#include <pcpsdefs.h>
#include <gpsdefs.h>
#include <mbg_tgt.h>
#include <use_pack.h>

#if defined( MBG_TGT_WIN32 )

  #if defined( _KDD_ )
    #include <ntddk.h>
  #else
    #include <windows.h>
  #endif

#elif defined( MBG_TGT_LINUX )

  //##++ #include <asm/timex.h>

#endif



/* Start of header body */

#if defined( _USE_PACK )   // set byte alignment
  #pragma pack( 1 )
#endif


#if defined( MBG_TGT_WIN32 )

  typedef LARGE_INTEGER MBG_PC_CYCLES;

#elif defined( MBG_TGT_LINUX )

  typedef uint64_t MBG_PC_CYCLES;

#elif defined( MBG_TGT_OS2 )

  typedef uint32_t MBG_PC_CYCLES;  //##++ should differentiate more

#elif defined( MBG_TGT_DOS )

  typedef uint32_t MBG_PC_CYCLES;  //##++ should differentiate more

#else // other target OSs which access the hardware directly

  typedef uint32_t MBG_PC_CYCLES;  //##++ should differentiate more

#endif


typedef uint8_t PCPS_STATUS_PORT;
typedef uint32_t PCPS_ASIC_VERSION;


// ------ definitions used when accessing a clock -------------------
//  (also refer to pcpsdefs.h for more common definitions)

// Bit masks of the board's status port
#define PCPS_ST_BUSY  0x01  // the clock is busy filling the output FIFO
#define PCPS_ST_IRQF  0x02  // the clock has generated an IRQ on the PC bus
#define PCPS_ST_MOD   0x20  // the DCF77 modulation
#define PCPS_ST_SEC   0x40  // Seconds have changed since last reading
#define PCPS_ST_MIN   0x80  // Minutes have changed since last reading

// The flags PCPS_ST_SEC and PCPS_ST_MIN are cleared whenever the clock
// is read, so they are very unreliable in multitasking environments
// and should therefore be ignored.



// The following flags describe the bus types which are
// supported by the plugin clocks.
#define PCPS_BUS_ISA        0x0001    // IBM compatible PC/AT ISA bus
#define PCPS_BUS_MCA        0x0002    // IBM PS/2 micro channel
#define PCPS_BUS_PCI        0x0004    // PCI

// The flag below is set if a clock with PCI interface uses
// the S5920 chip instead of the S5933.
#define PCPS_BUS_PCI_S5920  0x8000
#define PCPS_BUS_PCI_ASIC   0x4000

// The constant below combines the PCI bus flags.
#define PCPS_BUS_PCI2       ( PCPS_BUS_PCI | PCPS_BUS_PCI_S5920 )
#define PCPS_BUS_PCI3       ( PCPS_BUS_PCI | PCPS_BUS_PCI_ASIC )



// A list of known radio clocks.
enum PCPS_TYPES
{
  PCPS_TYPE_PC31,
  PCPS_TYPE_PS31_OLD,
  PCPS_TYPE_PS31,
  PCPS_TYPE_PC32,
  PCPS_TYPE_PCI32,
  PCPS_TYPE_GPS167PC,
  PCPS_TYPE_GPS167PCI,
  PCPS_TYPE_PCI509,
  PCPS_TYPE_GPS168PCI,
  PCPS_TYPE_PCI510,
  PCPS_TYPE_GPS169PCI,
  PCPS_TYPE_TCR510PCI,
  PCPS_TYPE_TCR167PCI,
  N_PCPS_DEV_TYPE
};


// The structure below contains the characteristics of each
// of the clocks listed above. These fields are always the
// same for a single type of clock and do not change with
// firmware version, port address, etc.

#define PCPS_CLOCK_NAME_SZ   10   // including terminating 0

typedef uint16_t PCPS_BUS_FLAGS;

typedef struct
{
  uint16_t num;
  char name[PCPS_CLOCK_NAME_SZ];
  uint16_t dev_id;
  uint16_t ref_type;
  PCPS_BUS_FLAGS bus_flags;
} PCPS_DEV_TYPE;



// The structure below describes an I/O port resource
// used by a clock.
typedef struct
{
  uint16_t base;
  uint16_t num;
} PCPS_PORT_RSRC;

// The max number of I/O port resources used by a clock.
#define N_PCPS_PORT_RSRC 2


// The structure below contains data which depends
// on a individual instance of the clock, e.g.
// the firmware which is currently installed, the
// port address which has been configured, etc.
typedef uint32_t PCPS_ERR_FLAGS;
typedef uint32_t PCPS_FEATURES;
typedef uint16_t PCPS_BUS_NUM;
typedef uint16_t PCPS_SLOT_NUM;

typedef struct
{
  PCPS_ERR_FLAGS err_flags;
  PCPS_BUS_NUM bus_num;
  PCPS_SLOT_NUM slot_num;
  PCPS_PORT_RSRC port[N_PCPS_PORT_RSRC];
  uint16_t status_port;
  uint16_t irq_num;
  uint32_t timeout_clk;
  uint16_t fw_rev_num;
  PCPS_FEATURES features;
  PCPS_ID_STR fw_id;
  PCPS_SN_STR sernum;
} PCPS_DEV_CFG;

// Flags used with PCPS_DEV_CFG.err_flags:
#define PCPS_EF_TIMEOUT         0x00000001   // timeout occured
#define PCPS_EF_INV_EPROM_ID    0x00000002   // invalid EPROM ID
#define PCPS_EF_IO_INIT         0x00000004   // I/O intf not init'd
#define PCPS_EF_IO_CFG          0x00000008   // I/O intf not cfg'd
#define PCPS_EF_IO_ENB          0x00000010   // I/O intf not enabled
#define PCPS_EF_IO_RSRC         0x00000020   // I/O not registered w/ rsrcmgr

// Some features of the radio clocks have been introduced with
// specific firmware versions, so depending on the firmware version
// a clock may support a feature or not. The clock detection function
// checks the clock model and firmware version and updates the field
// PCPS_DEV_CFG.features accordingly. There are some macros which
// can easily be used to query whether a clock device actually
// supports a function, or not. The definitions below define
// the possible features:
#define PCPS_CAN_SET_TIME       0x00000001UL
#define PCPS_HAS_SERIAL         0x00000002UL
#define PCPS_HAS_SYNC_TIME      0x00000004UL
#define PCPS_HAS_TZDL           0x00000008UL
#define PCPS_HAS_IDENT          0x00000010UL
#define PCPS_HAS_UTC_OFFS       0x00000020UL
#define PCPS_HAS_HR_TIME        0x00000040UL
#define PCPS_HAS_SERNUM         0x00000080UL
#define PCPS_HAS_TZCODE         0x00000100UL
#define PCPS_HAS_CABLE_LEN      0x00000200UL
#define PCPS_HAS_EVENT_TIME     0x00000400UL  // custom GPS firmware only
#define PCPS_HAS_RECEIVER_INFO  0x00000800UL
#define PCPS_CAN_CLR_UCAP_BUFF  0x00001000UL
#define PCPS_HAS_PCPS_TZDL      0x00002000UL
#define PCPS_HAS_UCAP           0x00004000UL
#define PCPS_HAS_IRIG_TX        0x00008000UL
#define PCPS_HAS_GPS_DATA_16    0x00010000UL  // use 16 bit size specifiers

// The constants below define those features which are available
// in ALL firmware versions which have been shipped with a
// specific clock.

#define PCPS_FEAT_PC31PS31  0

// Some of the features are available in all newer clocks,
// so these have been put together in one definition:
#define PCPS_FEAT_LVL2      ( PCPS_CAN_SET_TIME   \
                            | PCPS_HAS_SERIAL     \
                            | PCPS_HAS_SYNC_TIME  \
                            | PCPS_HAS_UTC_OFFS )

#define PCPS_FEAT_PC32      ( PCPS_FEAT_LVL2 )

#define PCPS_FEAT_PCI32     ( PCPS_FEAT_LVL2 )

#define PCPS_FEAT_PCI509    ( PCPS_FEAT_LVL2 \
                            | PCPS_HAS_SERNUM \
                            | PCPS_HAS_TZCODE )

#define PCPS_FEAT_PCI510    ( PCPS_FEAT_PCI509 )

#define PCPS_FEAT_GPS167PC  ( PCPS_FEAT_LVL2 \
                            | PCPS_HAS_TZDL \
                            | PCPS_HAS_IDENT )

#define PCPS_FEAT_GPS167PCI ( PCPS_FEAT_LVL2 \
                            | PCPS_HAS_TZDL  \
                            | PCPS_HAS_IDENT \
                            | PCPS_HAS_HR_TIME )

#define PCPS_FEAT_GPS168PCI ( PCPS_FEAT_LVL2 \
                            | PCPS_HAS_TZDL  \
                            | PCPS_HAS_IDENT \
                            | PCPS_HAS_HR_TIME \
                            | PCPS_HAS_CABLE_LEN \
                            | PCPS_HAS_RECEIVER_INFO )

#define PCPS_FEAT_GPS169PCI ( PCPS_FEAT_GPS168PCI \
                            | PCPS_CAN_CLR_UCAP_BUFF \
                            | PCPS_HAS_UCAP )

#define PCPS_FEAT_TCR510PCI ( PCPS_FEAT_LVL2 \
                            | PCPS_HAS_SERNUM )

#define PCPS_FEAT_TCR167PCI ( PCPS_FEAT_LVL2 \
                            | PCPS_HAS_SERNUM \
                            | PCPS_HAS_TZDL \
                            | PCPS_HAS_HR_TIME \
                            | PCPS_HAS_RECEIVER_INFO \
                            | PCPS_CAN_CLR_UCAP_BUFF \
                            | PCPS_HAS_UCAP \
                            | PCPS_HAS_IRIG_TX \
                            | PCPS_HAS_GPS_DATA_16 )


// Some features of the API used to access Meinberg plug-in radio clocks
// have been implemented starting with the special firmware revision
// numbers defined below.
//
// If no number is specified for a feature/clock model then the feature
// is either always supported by that clock model, or not at all.


/* This board uses the GPS_DATA interface with 16 bit buffer sizes 
   instead of the original 8 bit sizes, thus allowing to transfer 
   data blocks which exceed 255 bytes (PCPS_HAS_GPS_DATA_16) */
#define REV_HAS_GPS_DATA_16_GPS169PCI    0x0202

/* the clock supports a higher baud rate than N_PCPS_BD_DCF */
#define REV_HAS_SERIAL_HS_PCI509         0x0104

/* commands PCPS_GIVE_UCAP_ENTRIES, PCPS_GIVE_UCAP_EVENT */
#define REV_HAS_UCAP_GPS167PCI           0x0421
#define REV_HAS_UCAP_GPS168PCI           0x0104

/* command PCPS_CLR_UCAP_BUFF */
#define REV_CAN_CLR_UCAP_BUFF_GPS167PCI  0x0419
#define REV_CAN_CLR_UCAP_BUFF_GPS168PCI  0x0101

/* commands PCPS_READ_GPS_DATA and PCPS_WRITE_GPS_DATA with */
/* code PC_GPS_ANT_CABLE_LEN */
#define REV_HAS_CABLE_LEN_GPS167PCI      0x0411
#define REV_HAS_CABLE_LEN_GPS167PC       0x0411

/* command PCPS_GIVE_HR_TIME, structure PCPS_HR_TIME */
#define REV_HAS_HR_TIME_GPS167PC         0x0305
#define REV_HAS_HR_TIME_TCR510PCI        0x0200

/* field offs_utc in structure PCPS_TIME */
#define REV_HAS_UTC_OFFS_PC31PS31        0x0300

/* command PCPS_GIVE_SYNC_TIME: */
#define REV_HAS_SYNC_TIME_PC31PS31       0x0300

/* command PCPS_GET_SERIAL, PCPS_SET_SERIAL: */
#define REV_HAS_SERIAL_PC31PS31          0x0300

/* command PCPS_GIVE_TIME_NOCLEAR: */
#define REV_GIVE_TIME_NOCLEAR_PC31PS31   0x0300

/* status bit PCPS_LS_ANN: */
#define REV_PCPS_LS_ANN_PC31PS31         0x0300

/* status bit PCPS_IFTM: */
#define REV_PCPS_IFTM_PC31PS31           0x0300

/* command PCPS_SET_TIME: */
#define REV_CAN_SET_TIME_PC31PS31        0x0240

/* command PCPS_GIVE_TIME_NOCLEAR: */
// This is supported by all clocks but PC31/PS31 with
// firmware versions before v3.0. If such a card shall 
// be used then the firmware should be updated to the
// last recent version.


// The structure below has been defined to pass all
// information on a clock device from a device driver
// to a user program.
typedef struct
{
  PCPS_DEV_TYPE type;
  PCPS_DEV_CFG cfg;
} PCPS_DEV;


// The macros below simplify access to the data
// stored in PCPS_DEV structure and should be used
// to extract the desired information.
// If the formal parameter is called _d then a pointer
// to device structure PCPS_DEV is expected.
// If the formal parameter is called _c then a pointer
// to configuration structure PCPS_DEV_CFG is expected.

// Access device type information:
#define _pcps_type_num( _d )     ( (_d)->type.num )
#define _pcps_type_name( _d )    ( (_d)->type.name )
#define _pcps_dev_id( _d )       ( (_d)->type.dev_id )
#define _pcps_ref_type( _d )     ( (_d)->type.ref_type )
#define _pcps_bus_flags( _d )    ( (_d)->type.bus_flags )

// Query device type features:
#define _pcps_is_gps( _d )       ( _pcps_ref_type( _d ) == PCPS_REF_GPS )
#define _pcps_is_dcf( _d )       ( _pcps_ref_type( _d ) == PCPS_REF_DCF )
#define _pcps_is_irig_rx( _d )   ( _pcps_ref_type( _d ) == PCPS_REF_IRIG )

#define _pcps_is_isa( _d )       ( _pcps_bus_flags( _d ) & PCPS_BUS_ISA )
#define _pcps_is_mca( _d )       ( _pcps_bus_flags( _d ) & PCPS_BUS_MCA )
#define _pcps_is_pci( _d )       ( _pcps_bus_flags( _d ) & PCPS_BUS_PCI )
#define _pcps_is_pci_s5920( _d ) ( _pcps_bus_flags( _d ) & PCPS_BUS_PCI_S5920 )
#define _pcps_is_pci_asic( _d )  ( _pcps_bus_flags( _d ) & PCPS_BUS_PCI_ASIC )


// Access device configuration information:
#define _pcps_bus_num( _d )      ( (_d)->cfg.bus_num )
#define _pcps_slot_num( _d )     ( (_d)->cfg.slot_num )

#define _pcps_cfg_port_rsrc( _c, _n )  ( (_c)->port[_n] )
#define _pcps_port_rsrc( _d, _n )      _pcps_cfg_port_rsrc( &(_d)->cfg, (_n) )
#define _pcps_port_rsrc_unused( _d )   ( (_d)->base == 0 || (_d)->num == 0 )

#define _pcps_cfg_port_base( _c, _n )  ( _pcps_cfg_port_rsrc( (_c), (_n) ).base )
#define _pcps_port_base( _d, _n )      ( _pcps_port_rsrc( (_d), (_n) ).base )

#define _pcps_cfg_irq_num( _c )        ( (_c)->irq_num )
#define _pcps_irq_num( _d )            _pcps_cfg_irq_num( &(_d)->cfg )

#define _pcps_fw_rev_num( _d )         ( (_d)->cfg.fw_rev_num )
#define _pcps_fw_id( _d )              ( (_d)->cfg.fw_id )
#define _pcps_sernum( _d )             ( (_d)->cfg.sernum )


// The macros below handle the clock device's err_flags.
#define _pcps_err_flags( _d )           ( (_d)->cfg.err_flags )
#define _pcps_chk_err_flags( _d, _msk ) ( _pcps_err_flags( _d ) & (_msk) )
#define _pcps_set_err_flags( _d, _msk ) ( _pcps_err_flags( _d ) |= (_msk) )
#define _pcps_clr_err_flags( _d, _msk ) ( _pcps_err_flags( _d ) &= ~(_msk) )


// Query whether a special feature is supported:
#define _pcps_has_feature( _d, _f ) ( ( (_d)->cfg.features & (_f) ) != 0 )
#define _pcps_can_set_time( _d )   _pcps_has_feature( (_d), PCPS_CAN_SET_TIME )
#define _pcps_has_serial( _d )     _pcps_has_feature( (_d), PCPS_HAS_SERIAL )
#define _pcps_has_sync_time( _d )  _pcps_has_feature( (_d), PCPS_HAS_SYNC_TIME )
#define _pcps_has_ident( _d )      _pcps_has_feature( (_d), PCPS_HAS_IDENT )
#define _pcps_has_utc_offs( _d )   _pcps_has_feature( (_d), PCPS_HAS_UTC_OFFS )
#define _pcps_has_hr_time( _d )    _pcps_has_feature( (_d), PCPS_HAS_HR_TIME )
#define _pcps_has_sernum( _d )     _pcps_has_feature( (_d), PCPS_HAS_SERNUM )
#define _pcps_has_cab_len( _d )    _pcps_has_feature( (_d), PCPS_HAS_CABLE_LEN )
#define _pcps_has_tzdl( _d )       _pcps_has_feature( (_d), PCPS_HAS_TZDL )
#define _pcps_has_pcps_tzdl( _d )  _pcps_has_feature( (_d), PCPS_HAS_PCPS_TZDL )
#define _pcps_has_tzcode( _d )     _pcps_has_feature( (_d), PCPS_HAS_TZCODE )
#define _pcps_has_tz( _d )         _pcps_has_feature( (_d), PCPS_HAS_TZDL \
                                                          | PCPS_HAS_PCPS_TZDL \
                                                          | PCPS_HAS_TZCODE )
// The next one is supported only with a certain GPS firmware version:
#define _pcps_has_event_time( _d ) _pcps_has_feature( (_d), PCPS_HAS_EVENT_TIME )
#define _pcps_has_receiver_info( _d ) _pcps_has_feature( (_d), PCPS_HAS_RECEIVER_INFO )
#define _pcps_can_clr_ucap_buff( _d ) _pcps_has_feature( (_d), PCPS_CAN_CLR_UCAP_BUFF )
#define _pcps_has_ucap( _d )       _pcps_has_feature( (_d), PCPS_HAS_UCAP )
#define _pcps_has_irig_tx( _d )    _pcps_has_feature( (_d), PCPS_HAS_IRIG_TX )

// The macro below determines whether a DCF77 clock
// supports a higher baud rate than standard
#define _pcps_has_serial_hs( _d ) \
  ( ( _pcps_type_num( _d ) == PCPS_TYPE_TCR510PCI ) || \
    ( _pcps_type_num( _d ) == PCPS_TYPE_PCI510 ) || \
    ( _pcps_type_num( _d ) == PCPS_TYPE_PCI509  && \
      _pcps_fw_rev_num( _d ) >= REV_HAS_SERIAL_HS_PCI509 ) )


#define _pcps_has_signal( _d ) \
  ( _pcps_is_dcf( _d ) || _pcps_is_irig_rx( _d ) )

#define _pcps_has_mod( _d ) \
  ( _pcps_is_dcf( _d ) )


#define _pcps_has_irig( _d ) \
  ( _pcps_is_irig_rx( _d ) || _pcps_has_irig_tx( _d ) )

#define _pcps_has_ref_offs( _d ) \
  _pcps_is_irig_rx( _d )

#define _pcps_ref_offs_out_of_range( _n ) \
  ( abs( _n ) > PCPS_REF_OFFS_MAX )

#define _pcps_has_opt_flags( _d ) \
  _pcps_is_irig_rx( _d )

#define _pcps_has_gps_data_16( _d )  _pcps_has_feature( (_d), PCPS_HAS_GPS_DATA_16 )

#define _pcps_has_gps_data( _d ) \
  ( _pcps_is_gps( _d ) || _pcps_has_gps_data_16( _d ) )


// The structure below is used to return info
// on the device driver.
typedef struct
{
  uint16_t ver_num;    // the device driver's version number
  uint16_t n_devs;     // the number of radio clocks handled by the driver
  PCPS_ID_STR id_str;  // the device driver's ID string
} PCPS_DRVR_INFO;



/*
 * The definitions and types below are used to collect
 * all configuration parameters of a clock's serial ports
 * that can be handled by this library:
 */

// The maximum number of clocks' serial ports and string types
// that can be handled by the configuration programs:
#define MAX_PARM_PORT        4
#define MAX_PARM_STR_TYPE    10

typedef PORT_INFO_IDX ALL_PORT_INFO[MAX_PARM_PORT];
typedef STR_TYPE_INFO_IDX ALL_STR_TYPE_INFO[MAX_PARM_STR_TYPE];

typedef struct
{
  ALL_PORT_INFO pii;
  ALL_STR_TYPE_INFO stii;
  PORT_PARM tmp_pp;

} RECEIVER_PORT_CFG;



// The macros below can be used to mark a PCPS_TIME variable
// as unread, i.e. its contents have not been read from the clock,
// and to check if such a variable is marked as unread.
#define _pcps_time_set_unread( _t );      { (_t)->sec = 0xFF; }
#define _pcps_time_is_read( _t )          ( (uchar) (_t)->sec != 0xFF )



// The structures below are used to read current time from
// a device, combined with an associated PC cycle counter value
// to compensate program execution time. The cycle counter value
// is usually derived from the PC CPU's TSC and the type is
// redefined to a common type, depending on the operating system.
// The cycle counter clock frequency usually corresponds to the
// PC CPU clock frequency.

typedef struct
{
  MBG_PC_CYCLES cycles;
  PCPS_TIME t;
} PCPS_TIME_CYCLES;


typedef struct
{
  MBG_PC_CYCLES cycles;
  PCPS_HR_TIME t;
} PCPS_HR_TIME_CYCLES;



// Return codes which may be returned by the functions.

#define PCPS_ERR_TIMEOUT     -20   // Timeout accessing the board
#define PCPS_ERR_FW_ID       -21   // Invalid firmware ID
#define PCPS_ERR_NBYTES      -22   // The number of parameter bytes
                                   //   passed to the board did not match
                                   //   the number of bytes expected.
#define PCPS_ERR_INV_TIME    -23   // The board's time is not valid
#define PCPS_ERR_FIFO        -24   // The board's FIFO is empty, though
                                   //   it shouldn't be
#define PCPS_ERR_NOT_READY   -25   // Board is temporary unable to respond
                                   //   (during initialization after RESET)
#define PCPS_ERR_INV_TYPE    -26   // Board did not recognize data type


#if defined( _USE_PACK )   // set default alignment
  #pragma pack()
#endif

/* End of header body */

#undef _ext

#endif  /* _PCPSDEV_H */

