
/**************************************************************************
 *
 *  $Id: mbgdevio.c 1.10 2004/04/14 09:39:17Z martin TEST $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Functions called from user space to access Meinberg device drivers.
 *
 * -----------------------------------------------------------------------
 *  $Log: mbgdevio.c $
 *  Revision 1.10  2004/04/14 09:39:17Z  martin
 *  Allow [g|s]et_irig_info() also for new devices with IRIG output.
 *  Use MBGDEVIO_COMPAT_VERSION to check version.
 *  Revision 1.9  2003/12/22 15:30:52Z  martin
 *  Added functions mbg_get_asic_version(), mbg_get_time_cycles(),
 *  and mbg_get_hr_time_cycles().
 *  Support higher baud rates for TCR510PCI and PCI510.
 *  Support PCPS_HR_TIME for TCR510PCI.
 *  API calls return ioctl results instead of success/-1.
 *  Moved some Win32 specific code to mbgsvctl DLL.
 *  Log Win32 ioctl errors to event log for debugging.
 *  Revision 1.8  2003/06/19 08:42:33Z  martin
 *  Renamed function mbg_clr_cap_buff() to mbg_clr_ucap_buff().
 *  New functions mbg_get_ucap_entries() and mbg_get_ucap_event().
 *  New function mbg_get_hr_ucap().
 *  New functions mbgdevio_get_version() and mbgdevio_check_version().
 *  New functions for generic read/write access.
 *  New functions mbg_get_pcps_tzdl() and mbg_set_pcps_tzdl().
 *  Fixed a bug passing the wrong command code to a 
 *  direct access target in mbg_get_sync_time().
 *  Return driver info for direct access targets.
 *  Include pcpsdrvr.h and pcps_dos.h, if applicable.
 *  For direct access targets, check if a function is supported
 *  before accessing the hardware.
 *  Use const parameter pointers if applicable.
 *  Changes due to renamed symbols/macros.
 *  Source code cleanup.
 *  Revision 1.7  2003/05/16 08:52:46  MARTIN
 *  Swap doubles inside API functions.
 *  Enhanced support for direct access targets.
 *  Removed obsolete code.
 *  Revision 1.6  2003/04/25 10:14:16  martin
 *  Renamed macros.
 *  Extended macro calls for direct access targets.
 *  Updated macros for Linux.
 *  Revision 1.5  2003/04/15 19:35:25Z  martin
 *  New functions mbg_setup_receiver_info(), 
 *  mbg_get_serial_settings(), mbg_save_serial_settings().
 *  Revision 1.4  2003/04/09 16:07:16Z  martin
 *  New API functions mostly complete.
 *  Use renamed IOCTL codes from mbgioctl.h.
 *  Added DllEntry function foe Win32.
 *  Made MBG_Device_count and MBG_Device_Path local.
 *  Revision 1.3  2003/01/24 13:44:40Z  martin
 *  Fixed get_ref_time_from_driver_at_sec_change() to be used 
 *  with old kernel drivers.
 *  Revision 1.2  2002/09/06 11:04:01Z  martin
 *  Some old API functions have been replaced by new ones
 *  for a common PnP/non-PnP API.
 *  New API function which clears capture buffer.
 *  New function get_ref_time_from_driver_at_sec_change().
 *  Revision 1.1  2002/02/19 13:48:20Z  MARTIN
 *  Initial revision
 *
 **************************************************************************/

#define _MBGDEVIO
 #include <mbgdevio.h>
#undef _MBGDEVIO

#include <parmpcps.h>
#include <parmgps.h>
#include <gpsutils.h>


#if defined( MBG_USE_KERNEL_DRIVER )

  #include <mbgioctl.h>

#else

  #include <pcpsdrvr.h>
  #include <pci_asic.h>

  static PCPS_DRVR_INFO drvr_info = { MBGDEVIO_VERSION, 0, "MBGDEVIO direct" };

#endif



#if defined( MBG_TGT_WIN32 )

  #include <mbgsvctl.h>
  #include <tchar.h>
  #include <stdio.h>

  #define _mbgdevio_vars() \
    DWORD ReturnedLength;  \
    DWORD rc

  #define _mbgdevio_ret_val  rc

  #define _mbgdevio_read( _dh, _cmd, _ioctl, _p, _sz ) \
  {                                                    \
    if ( DeviceIoControl(                              \
           _dh, _ioctl,                                \
           NULL, 0,                                    \
           _p, _sz,                                    \
           &ReturnedLength,                            \
           NULL                                        \
         )                                             \
       )                                               \
    {                                                  \
      rc = PCPS_SUCCESS;                               \
    }                                                  \
    else                                               \
    {                                                  \
      rc = GetLastError();                             \
      mbgdevio_log_error( _ioctl, rc );                \
    }                                                  \
  }

  #define _mbgdevio_read_var( _dh, _cmd, _ioctl, _p ) \
    _mbgdevio_read( _dh, _cmd, _ioctl, _p, sizeof( *(_p) ) )

  #define _mbgdevio_write_var( _dh, _cmd, _ioctl, _p ) \
  {                                                    \
    if ( DeviceIoControl(                              \
           _dh, _ioctl,                                \
           (OUT LPVOID) _p, sizeof( *(_p) ),           \
           NULL, 0,                                    \
           &ReturnedLength,                            \
           NULL                                        \
         )                                             \
       )                                               \
    {                                                  \
      rc = PCPS_SUCCESS;                               \
    }                                                  \
    else                                               \
    {                                                  \
      rc = GetLastError();                             \
      mbgdevio_log_error( _ioctl, rc );                \
    }                                                  \
  }

  #define _mbgdevio_write_cmd( _dh, _cmd, _ioctl ) \
  {                                                \
    if ( DeviceIoControl(                          \
           _dh, _ioctl,                            \
           NULL, 0,                                \
           NULL, 0,                                \
           &ReturnedLength,                        \
           NULL                                    \
         )                                         \
       )                                           \
    {                                              \
      rc = PCPS_SUCCESS;                           \
    }                                              \
    else                                           \
    {                                              \
      rc = GetLastError();                         \
      mbgdevio_log_error( _ioctl, rc );            \
    }                                              \
  }

  #define _mbgdevio_read_gps        _mbgdevio_read

  #define _mbgdevio_read_gps_var    _mbgdevio_read_var

  #define _mbgdevio_write_gps_var   _mbgdevio_write_var

#elif defined( MBG_TGT_LINUX )

  #include <mbgioctl.h>

  #include <unistd.h>
  #include <stdlib.h>
  #include <string.h>
  #include <fcntl.h>
  #include <sys/ioctl.h>

  #define _mbgdevio_vars() \
    int rc

  #define _mbgdevio_ret_val \
    ( ( rc < 0 ) ? rc : PCPS_SUCCESS )

  #define _mbgdevio_read( _dh, _cmd, _ioctl, _p, _sz ) \
    rc = ioctl( _dh, _ioctl, _p )

  #define _mbgdevio_read_var( _dh, _cmd, _ioctl, _p ) \
    _mbgdevio_read( _dh, _cmd, _ioctl, _p, sizeof( *(_p) ) )

  #define _mbgdevio_write_var       _mbgdevio_read_var

  #define _mbgdevio_write_cmd( _dh, _cmd, _ioctl ) \
    rc = ioctl( _dh, _ioctl, NULL )

  #define _mbgdevio_read_gps        _mbgdevio_read

  #define _mbgdevio_read_gps_var    _mbgdevio_read_var

  #define _mbgdevio_write_gps_var   _mbgdevio_write_var

#else // other target OSs which access the hardware directly

  #if defined MBG_TGT_QNX_NTO
    #include <stdio.h>
    #include <sys/neutrino.h>
  #endif

  #if MBG_USE_DOS_TSR
    #include <pcps_dos.h>
  #else
    #define pcps_read_safe            _pcps_read
    #define pcps_write_safe           pcps_write
    #define pcps_read_gps_safe        pcps_read_gps
    #define pcps_write_gps_safe       pcps_write_gps

    #define _pcps_write_byte_safe     _pcps_write_byte
    #define _pcps_read_var_safe       _pcps_read_var
    #define _pcps_write_var_safe      _pcps_write_var
    #define _pcps_read_gps_var_safe   _pcps_read_gps_var
    #define _pcps_write_gps_var_safe  _pcps_write_gps_var
  #endif


  #define _mbgdevio_vars()   \
    int rc

  #define _mbgdevio_ret_val  rc

  #define _mbgdevio_read( _dh, _cmd, _ioctl, _p, _sz ) \
          rc = pcps_read_safe( _dh, _cmd, _p, _sz )

  #define _mbgdevio_read_var( _dh, _cmd, _ioctl, _p ) \
          rc = _pcps_read_var_safe( _dh, _cmd, *(_p) )

  #define _mbgdevio_write( _dh, _cmd, _ioctl, _p, _sz ) \
          rc = pcps_write_safe( _dh, _cmd, _p, _sz )

  #define _mbgdevio_write_var( _dh, _cmd, _ioctl, _p ) \
          rc = _pcps_write_var_safe( _dh, _cmd, *(_p) )

  #define _mbgdevio_write_cmd( _dh, _cmd, _ioctl ) \
          rc = _pcps_write_byte_safe( _dh, _cmd );

  #define _mbgdevio_read_gps( _dh, _cmd, _ioctl, _p, _sz ) \
          rc = pcps_read_gps_safe( _dh, _cmd, _p, _sz )

  #define _mbgdevio_read_gps_var( _dh, _cmd, _ioctl, _p ) \
          rc = _pcps_read_gps_var_safe( _dh, _cmd, *(_p) )

  #define _mbgdevio_write_gps( _dh, _cmd, _ioctl, _p, _sz ) \
          rc = pcps_write_gps_safe( _dh, _cmd, _p, _sz )

  #define _mbgdevio_write_gps_var( _dh, _cmd, _ioctl, _p ) \
          rc = _pcps_write_gps_var_safe( _dh, _cmd, *(_p) )

  #define _mbgdevio_chk_cond( _cond ) \
  {                                   \
    if ( !(_cond) )                   \
      return -1;                      \
  }                                   \

#endif  // target specific definitions



#if !defined( _mbgdevio_chk_cond )
  #define _mbgdevio_chk_cond( _cond );
#endif


#define _mbgdevio_read_chk( _dh, _cmd, _ioctl, _p, _sz, _cond ) \
{                                                               \
  _mbgdevio_chk_cond( _cond );                                  \
  _mbgdevio_read( _dh, _cmd, _ioctl, _p, _sz );                 \
}

#define _mbgdevio_read_var_chk( _dh, _cmd, _ioctl, _p, _cond ) \
{                                                              \
  _mbgdevio_chk_cond( _cond );                                 \
  _mbgdevio_read_var( _dh, _cmd, _ioctl, _p );                 \
}

#define _mbgdevio_write_var_chk( _dh, _cmd, _ioctl, _p, _cond ) \
{                                                               \
  _mbgdevio_chk_cond( _cond );                                  \
  _mbgdevio_write_var( _dh, _cmd, _ioctl, _p );                 \
}

#define _mbgdevio_write_cmd_chk( _dh, _cmd, _ioctl, _cond ) \
{                                                           \
  _mbgdevio_chk_cond( _cond );                              \
  _mbgdevio_write_cmd( _dh, _cmd, _ioctl );                 \
}

#define _mbgdevio_read_gps_chk( _dh, _cmd, _ioctl, _p, _sz, _cond ) \
{                                                                   \
  _mbgdevio_chk_cond( _cond );                                      \
  _mbgdevio_read_gps( _dh, _cmd, _ioctl, _p, _sz );                 \
}

#define _mbgdevio_read_gps_var_chk( _dh, _cmd, _ioctl, _p, _cond ) \
{                                                                  \
  _mbgdevio_chk_cond( _cond );                                     \
  _mbgdevio_read_gps_var( _dh, _cmd, _ioctl, _p );                 \
}

#define _mbgdevio_write_gps_var_chk( _dh, _cmd, _ioctl, _p, _cond ) \
{                                                                   \
  _mbgdevio_chk_cond( _cond );                                      \
  _mbgdevio_write_gps_var( _dh, _cmd, _ioctl, _p );                 \
}



#if defined( MBG_TGT_WIN32 )

static /*HDR*/
void _MBG_API mbgdevio_log_error( DWORD ioctl, DWORD rc )
{
  #define MBG_SVC_NAME_MBGDEVIO  "mbgdevio"

  #define MBG_CTL_CODE_IDX( _ioctl ) \
    ( ( ( (_ioctl) & 0x3FFF ) >> 2 ) - 0x930 )

  TCHAR szMsg[512];
  LPTSTR lpszStrings[1];
  HANDLE hEventSource = RegisterEventSource( NULL, TEXT( MBG_SVC_NAME_MBGDEVIO ) );


  if ( hEventSource != NULL ) 
  {
    WORD evt_type;
    int n = 0;
    LPVOID lpMsgBuf = NULL;

    if ( rc != ERROR_SUCCESS )
    {
      n = FormatMessage(
        FORMAT_MESSAGE_ALLOCATE_BUFFER |             // source and processing options
        FORMAT_MESSAGE_FROM_SYSTEM |                 //
        FORMAT_MESSAGE_IGNORE_INSERTS,               //
        NULL,                                        // msg source
        rc,                                          // msg identifier
        0, // MAKELANGID( LANG_NEUTRAL, SUBLANG_DEFAULT ), // language identifier
        (LPTSTR) &lpMsgBuf,                          // msg buffer
        0,                                           // max size of msg buffer
        NULL                                         // array of msg inserts
      );

      // Process any inserts in lpMsgBuf here ...
    }

    _stprintf( szMsg, TEXT( "ioctl 0x%02X, err %d: %s" ), 
               MBG_CTL_CODE_IDX( ioctl ), 
               rc,
               ( lpMsgBuf ) ? lpMsgBuf : "??"
             );

    // Free the buffer.
    if ( lpMsgBuf )
      LocalFree( lpMsgBuf );

    
    lpszStrings[0] = szMsg;

    if ( rc == ERROR_SUCCESS )
      evt_type = EVENTLOG_INFORMATION_TYPE;
    else
      if ( rc == ERROR_NOACCESS )
        evt_type = EVENTLOG_WARNING_TYPE;
      else
        evt_type = EVENTLOG_ERROR_TYPE;

     ReportEvent( hEventSource,         // handle of event source
                  evt_type,             // event type
                  0,                    // event category
                  0,                    // event ID
                  NULL,                 // current user's SID
                  1,                    // strings in lpszStrings
                  0,                    // no bytes of raw data
                  lpszStrings,          // array of error strings
                  NULL );               // no raw data

    DeregisterEventSource( hEventSource );
  }

}  // mbgdevio_log_error

#endif  // defined( MBG_TGT_WIN32 )



/*HDR*/
_MBG_API_ATTR int _MBG_API mbgdevio_get_version( void )
{

  return MBGDEVIO_VERSION;

}  // mbgdevio_get_version



/*HDR*/
_MBG_API_ATTR int _MBG_API mbgdevio_check_version( int header_version )
{
  if ( header_version >= MBGDEVIO_COMPAT_VERSION )
    return PCPS_SUCCESS;

  return -1;

}  // mbgdevio_check_version



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_find_devices( void )
{
  #if defined( _PCPSDRVR_H )

    #if defined MBG_TGT_QNX_NTO
      // Since this program accessed the hardware directly
      // I/O privileges must be assigned to the thread.
      if ( ThreadCtl( _NTO_TCTL_IO, NULL ) == -1 )
      {
        perror( "Fatal error" );
        exit( 1 );
      }
    #endif

    pcps_detect_clocks( pcps_isa_ports, NULL );

    return n_ddevs;

  #elif defined( MBG_TGT_WIN32 )

    return mbg_svc_find_devices();

  #elif defined( MBG_TGT_LINUX )

    MBG_DEV_HANDLE dh = mbg_open_device( 0 );

    if ( dh != MBG_INVALID_DEV_HANDLE )
    {
      mbg_close_device( &dh );
      return 1;
    }

    return 0;

  #endif

}  // mbg_find_devices



/*HDR*/
_MBG_API_ATTR MBG_DEV_HANDLE _MBG_API mbg_open_device( unsigned int device_index )
{
#if defined( MBG_TGT_WIN32 )

  const char *device_path;
  HANDLE dh;

  device_path = mbg_svc_get_device_path( device_index );

  if ( device_path == NULL )
    goto fail;


  dh = CreateFile( 
         device_path,                   // file name
         GENERIC_READ | GENERIC_WRITE,  // access mode
         0,                             // share mode
         NULL,                          // security descriptor
         OPEN_EXISTING,                 // how to create
         0,                             // file attributes
         NULL                           // handle to template file
       );

  if ( INVALID_HANDLE_VALUE == dh )
  {
    #if 0 //##++
      printf( "mbg_open_device: CreateFile failed for index %i\n", device_index );
    #endif

    goto fail;
  }

  return dh;

fail:  
  return MBG_INVALID_DEV_HANDLE;

#elif defined ( MBG_TGT_LINUX )

  static const char *dev_fn = "/dev/mbgclk";    //##++

  MBG_DEV_HANDLE dh = open( dev_fn, 0 );

  return ( dh < 0 ) ? MBG_INVALID_DEV_HANDLE : dh;

#else

  return ( device_index < n_ddevs ) ? &pcps_ddev[device_index] : NULL;

#endif

} // mbg_open_device



/*HDR*/
_MBG_API_ATTR void _MBG_API mbg_close_device( MBG_DEV_HANDLE *dev_handle )
{
#if defined( MBG_TGT_WIN32 )

  if ( *dev_handle != MBG_INVALID_DEV_HANDLE )
    CloseHandle( *dev_handle );

#elif defined ( MBG_TGT_LINUX )

  close( *dev_handle );

#endif

 *dev_handle = MBG_INVALID_DEV_HANDLE;

}  // mbg_close_device



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_drvr_info( MBG_DEV_HANDLE dh, PCPS_DRVR_INFO *p )
{
  #if defined( _MBGIOCTL_H )
    _mbgdevio_vars();
    _mbgdevio_read_var( dh, -1, IOCTL_GET_PCPS_DRVR_INFO, p );
    return _mbgdevio_ret_val;
  #else
    #if defined( __BORLANDC__ )
      dh;   // avoid warnings "never used"
    #endif
    drvr_info.n_devs = n_ddevs;
    *p = drvr_info;
    return PCPS_SUCCESS;
  #endif
}  // mbg_get_drvr_info



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_device_info( MBG_DEV_HANDLE dh, PCPS_DEV *p )
{
  #if defined( _MBGIOCTL_H )
    _mbgdevio_vars();
    _mbgdevio_read_var( dh, -1, IOCTL_GET_PCPS_DEV, p );
    return _mbgdevio_ret_val;
  #else
    *p = dh->dev;
    return PCPS_SUCCESS;
  #endif

}  // mbg_get_device_info



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_status_port( MBG_DEV_HANDLE dh, uchar *p )
{
  #if defined( _MBGIOCTL_H )
    _mbgdevio_vars();
    _mbgdevio_read_var( dh, -1, IOCTL_GET_PCPS_STATUS_PORT, p );
    return _mbgdevio_ret_val;
  #else
    *p = _pcps_ddev_read_status_port( dh );
    return PCPS_SUCCESS;
  #endif

}  // mbg_get_status_port



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_generic_read( MBG_DEV_HANDLE dh, int cmd,
                                             void *p, int size )
{
  #if defined( _MBGIOCTL_H )
    return -1;   // not yet implemented
  #else
    _mbgdevio_vars();
    _mbgdevio_read( dh, cmd, -1, p, size );
    return _mbgdevio_ret_val;
  #endif

}  // mbg_generic_read



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_generic_write( MBG_DEV_HANDLE dh, int cmd,
                                              const void *p, int size )
{
  #if defined( _MBGIOCTL_H )
    return -1;   // not yet implemented
  #else
    _mbgdevio_vars();
    _mbgdevio_write( dh, cmd, -1, p, size );
    return _mbgdevio_ret_val;
  #endif

}  // mbg_generic_write



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_generic_read_gps( MBG_DEV_HANDLE dh, int cmd,
                                                 void *p, int size )
{
  #if defined( _MBGIOCTL_H )
    return -1;   // not yet implemented
  #else
    _mbgdevio_vars();
    _mbgdevio_read_gps( dh, cmd, -1, p, size );
    return _mbgdevio_ret_val;
  #endif

}  // mbg_generic_read_gps



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_generic_write_gps( MBG_DEV_HANDLE dh, int cmd,
                                                  const void *p, int size )
{
  #if defined( _MBGIOCTL_H )
    return -1;   // not yet implemented
  #else
    _mbgdevio_vars();
    _mbgdevio_write_gps( dh, cmd, -1, p, size );
    return _mbgdevio_ret_val;
  #endif

}  // mbg_generic_write_gps



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_time( MBG_DEV_HANDLE dh, PCPS_TIME *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_var( dh, PCPS_GIVE_TIME, IOCTL_GET_PCPS_TIME, p );
  return _mbgdevio_ret_val;

}  // mbg_get_time



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_time( MBG_DEV_HANDLE dh, const PCPS_STIME *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_var_chk( dh, PCPS_SET_TIME, IOCTL_SET_PCPS_TIME, p,
                           _pcps_ddev_can_set_time( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_set_time



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_sync_time( MBG_DEV_HANDLE dh, PCPS_TIME *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_var_chk( dh, PCPS_GIVE_SYNC_TIME, IOCTL_GET_PCPS_SYNC_TIME,
                          p, _pcps_ddev_has_sync_time( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_get_sync_time



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_time_sec_change( MBG_DEV_HANDLE dh, PCPS_TIME *p )
{
  #if defined MBG_TGT_WIN32
    _mbgdevio_vars();
    _mbgdevio_read_var( dh, -1, IOCTL_GET_PCPS_TIME_SEC_CHANGE, p );
    return _mbgdevio_ret_val;
  #else
    #if defined( __BORLANDC__ )
      dh; p;  // avoid warnings "never used"
    #endif
    return -1;
  #endif

}  // mbg_get_time_sec_change



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_hr_time( MBG_DEV_HANDLE dh, PCPS_HR_TIME *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_var_chk( dh, PCPS_GIVE_HR_TIME, IOCTL_GET_PCPS_HR_TIME,
                          p, _pcps_ddev_has_hr_time( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_get_hr_time



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_event_time( MBG_DEV_HANDLE dh, const PCPS_TIME_STAMP *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_var_chk( dh, PCPS_SET_EVENT_TIME, IOCTL_SET_PCPS_EVENT_TIME,
                           p, _pcps_ddev_has_event_time( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_set_event_time



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_serial( MBG_DEV_HANDLE dh, PCPS_SERIAL *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_var( dh, PCPS_GET_SERIAL, IOCTL_GET_PCPS_SERIAL, p );
  return _mbgdevio_ret_val;

}  // mbg_get_serial



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_serial( MBG_DEV_HANDLE dh, const PCPS_SERIAL *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_var( dh, PCPS_SET_SERIAL, IOCTL_SET_PCPS_SERIAL, p );
  return _mbgdevio_ret_val;

}  // mbg_set_serial



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_tzcode( MBG_DEV_HANDLE dh, PCPS_TZCODE *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_var_chk( dh, PCPS_GET_TZCODE, IOCTL_GET_PCPS_TZCODE,
                          p, _pcps_ddev_has_tzcode( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_get_tzcode



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_tzcode( MBG_DEV_HANDLE dh, const PCPS_TZCODE *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_var_chk( dh, PCPS_SET_TZCODE, IOCTL_SET_PCPS_TZCODE,
                           p, _pcps_ddev_has_tzcode( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_set_tzcode



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_pcps_tzdl( MBG_DEV_HANDLE dh, PCPS_TZDL *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_var_chk( dh, PCPS_GET_PCPS_TZDL, IOCTL_GET_PCPS_TZDL,
                          p, _pcps_ddev_has_pcps_tzdl( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_get_pcps_tzdl



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_pcps_tzdl( MBG_DEV_HANDLE dh, const PCPS_TZDL *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_var_chk( dh, PCPS_SET_PCPS_TZDL, IOCTL_SET_PCPS_TZDL,
                           p, _pcps_ddev_has_pcps_tzdl( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_set_pcps_tzdl



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_ref_offs( MBG_DEV_HANDLE dh, PCPS_REF_OFFS *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_var_chk( dh, PCPS_GET_REF_OFFS, IOCTL_GET_PCPS_REF_OFFS,
                          p, _pcps_ddev_has_ref_offs( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_get_ref_offs



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_ref_offs( MBG_DEV_HANDLE dh, const PCPS_REF_OFFS *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_var_chk( dh, PCPS_SET_REF_OFFS, IOCTL_SET_PCPS_REF_OFFS,
                           p, _pcps_ddev_has_ref_offs( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_set_ref_offs



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_opt_info( MBG_DEV_HANDLE dh, PCPS_OPT_INFO *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_var_chk( dh, PCPS_GET_OPT_INFO, IOCTL_GET_PCPS_OPT_INFO,
                          p, _pcps_ddev_has_opt_flags( dh ) );
 return _mbgdevio_ret_val;

}  // mbg_get_opt_info



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_opt_settings( MBG_DEV_HANDLE dh, const PCPS_OPT_SETTINGS *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_var_chk( dh, PCPS_SET_OPT_SETTINGS,
                           IOCTL_SET_PCPS_OPT_SETTINGS, p,
                           _pcps_ddev_has_opt_flags( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_set_opt_settings



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_irig_info( MBG_DEV_HANDLE dh, IRIG_INFO *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_var_chk( dh, PCPS_GET_IRIG_INFO, IOCTL_GET_PCPS_IRIG_INFO,
                          p, _pcps_ddev_has_irig( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_get_irig_info



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_irig_settings( MBG_DEV_HANDLE dh, const IRIG_SETTINGS *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_var_chk( dh, PCPS_SET_IRIG_SETTINGS,
                           IOCTL_SET_PCPS_IRIG_SETTINGS, p,
                           _pcps_ddev_has_irig( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_set_irig_settings



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_clr_ucap_buff( MBG_DEV_HANDLE dh )
{
  _mbgdevio_vars();
  _mbgdevio_write_cmd_chk( dh, PCPS_CLR_UCAP_BUFF, IOCTL_PCPS_CLR_UCAP_BUFF,
                           _pcps_ddev_can_clr_ucap_buff( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_clr_ucap_buff



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_ucap_entries( MBG_DEV_HANDLE dh, PCPS_UCAP_ENTRIES *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_var_chk( dh, PCPS_GIVE_UCAP_ENTRIES,
                          IOCTL_GET_PCPS_UCAP_ENTRIES, p,
                          _pcps_ddev_has_ucap( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_get_ucap_entries



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_ucap_event( MBG_DEV_HANDLE dh, PCPS_HR_TIME *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_var_chk( dh, PCPS_GIVE_UCAP_EVENT,
                          IOCTL_GET_PCPS_UCAP_EVENT, p,
                          _pcps_ddev_has_ucap( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_get_ucap_event



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_gps_tzdl( MBG_DEV_HANDLE dh, TZDL *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_gps_var( dh, PC_GPS_TZDL, IOCTL_GET_GPS_TZDL, p );
  return _mbgdevio_ret_val;

}  // mbg_get_gps_tzdl



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_gps_tzdl( MBG_DEV_HANDLE dh, const TZDL *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_gps_var( dh, PC_GPS_TZDL, IOCTL_SET_GPS_TZDL, p );
  return _mbgdevio_ret_val;

}  // mbg_set_gps_tzdl



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_gps_sw_rev( MBG_DEV_HANDLE dh, SW_REV *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_gps_var( dh, PC_GPS_SW_REV, IOCTL_GET_GPS_SW_REV, p );
  return _mbgdevio_ret_val;

}  // mbg_get_gps_sw_rev



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_gps_stat( MBG_DEV_HANDLE dh, ushort *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_gps_var( dh, PC_GPS_STAT, IOCTL_GET_GPS_STAT, p );
  return _mbgdevio_ret_val;

}  // mbg_get_gps_stat



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_gps_time( MBG_DEV_HANDLE dh, TTM *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_gps_var( dh, PC_GPS_TIME, IOCTL_GET_GPS_TIME, p );
  return _mbgdevio_ret_val;

}  // mbg_get_gps_time



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_gps_time( MBG_DEV_HANDLE dh, const TTM *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_gps_var( dh, PC_GPS_TIME, IOCTL_SET_GPS_TIME, p );
  return _mbgdevio_ret_val;

}  // mbg_set_gps_time



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_gps_port_parm( MBG_DEV_HANDLE dh, PORT_PARM *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_gps_var( dh, PC_GPS_PORT_PARM, IOCTL_GET_GPS_PORT_PARM, p );
  return _mbgdevio_ret_val;

}  // mbg_get_gps_port_parm



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_gps_port_parm( MBG_DEV_HANDLE dh, const PORT_PARM *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_gps_var( dh, PC_GPS_PORT_PARM, IOCTL_SET_GPS_PORT_PARM, p );
  return _mbgdevio_ret_val;

}  // mbg_set_gps_port_parm



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_gps_ant_info( MBG_DEV_HANDLE dh, ANT_INFO *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_gps_var( dh, PC_GPS_ANT_INFO, IOCTL_GET_GPS_ANT_INFO, p );
  return _mbgdevio_ret_val;

}  // mbg_get_gps_ant_info



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_gps_ucap( MBG_DEV_HANDLE dh, TTM *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_gps_var( dh, PC_GPS_UCAP, IOCTL_GET_GPS_UCAP, p );
  return _mbgdevio_ret_val;

}  // mbg_get_gps_ucap



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_gps_enable_flags( MBG_DEV_HANDLE dh, ENABLE_FLAGS *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_gps_var( dh, PC_GPS_ENABLE_FLAGS,
                          IOCTL_GET_GPS_ENABLE_FLAGS, p );
  return _mbgdevio_ret_val;

}  // mbg_get_gps_enable_flags



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_gps_enable_flags( MBG_DEV_HANDLE dh,
                                       const ENABLE_FLAGS *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_gps_var( dh, PC_GPS_ENABLE_FLAGS,
                           IOCTL_SET_GPS_ENABLE_FLAGS, p );
  return _mbgdevio_ret_val;

}  // mbg_set_gps_enable_flags



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_gps_stat_info( MBG_DEV_HANDLE dh, STAT_INFO *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_gps_var( dh, PC_GPS_STAT_INFO, IOCTL_GET_GPS_STAT_INFO, p );
  return _mbgdevio_ret_val;

}  // mbg_get_gps_stat_info



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_gps_cmd( MBG_DEV_HANDLE dh, const ushort *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_gps_var( dh, PC_GPS_CMD, IOCTL_SET_GPS_CMD, p );
  return _mbgdevio_ret_val;

}  // mbg_set_gps_cmd


/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_gps_pos( MBG_DEV_HANDLE dh, POS *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_gps_var( dh, PC_GPS_POS, IOCTL_GET_GPS_POS, p );
  swap_pos_doubles( p );
  return _mbgdevio_ret_val;

}  // mbg_get_gps_pos



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_gps_pos_xyz( MBG_DEV_HANDLE dh, const XYZ p )
{
  _mbgdevio_vars();
  XYZ xyz;
  int i;

  for ( i = 0; i < N_XYZ; i++ )
  {
    xyz[i] = p[i];
    swap_double( &xyz[i] );
  }

  _mbgdevio_write_gps_var( dh, PC_GPS_POS_XYZ, IOCTL_SET_GPS_POS_XYZ, xyz );
  return _mbgdevio_ret_val;

}  // mbg_set_gps_pos_xyz



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_gps_pos_lla( MBG_DEV_HANDLE dh, const LLA p )
{
  _mbgdevio_vars();
  LLA lla;
  int i;

  for ( i = 0; i < N_LLA; i++ )
  {
    lla[i] = p[i];
    swap_double( &lla[i] );
  }

  _mbgdevio_write_gps_var( dh, PC_GPS_POS_LLA, IOCTL_SET_GPS_POS_LLA, lla );
  return _mbgdevio_ret_val;

}  // mbg_set_gps_pos_lla



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_gps_ant_cable_len( MBG_DEV_HANDLE dh, ANT_CABLE_LEN *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_gps_var_chk( dh, PC_GPS_ANT_CABLE_LEN,
                              IOCTL_GET_GPS_ANT_CABLE_LEN, p,
                              _pcps_ddev_has_cab_len( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_get_gps_ant_cable_len



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_gps_ant_cable_len( MBG_DEV_HANDLE dh,
                                                      const ANT_CABLE_LEN *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_gps_var_chk( dh, PC_GPS_ANT_CABLE_LEN,
                               IOCTL_SET_GPS_ANT_CABLE_LEN, p,
                               _pcps_ddev_has_cab_len( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_set_gps_ant_cable_len



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_gps_receiver_info( MBG_DEV_HANDLE dh, RECEIVER_INFO *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_gps_var_chk( dh, PC_GPS_RECEIVER_INFO,
                              IOCTL_GET_GPS_RECEIVER_INFO, p,
                              _pcps_ddev_has_receiver_info( dh ) );
         
  return _mbgdevio_ret_val;

}  // mbg_get_gps_receiver_info



static /*HDR*/  //##++
int _MBG_API mbg_get_gps_all_str_type_info( MBG_DEV_HANDLE dh,
                                            STR_TYPE_INFO_IDX stii[],
                                            const RECEIVER_INFO *p_ri )
{
  _mbgdevio_vars(); //##++
  _mbgdevio_read_gps_chk( dh, PC_GPS_ALL_STR_TYPE_INFO,
                          IOCTL_GET_GPS_ALL_STR_TYPE_INFO, stii,
                          p_ri->n_str_type * sizeof( stii[0] ),
                          _pcps_ddev_has_receiver_info( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_get_gps_all_str_type_info



static /*HDR*/  //##++
int _MBG_API mbg_get_gps_all_port_info( MBG_DEV_HANDLE dh,
                                        PORT_INFO_IDX pii[],
                                        const RECEIVER_INFO *p_ri )
{
  _mbgdevio_vars();
  _mbgdevio_read_gps_chk( dh, PC_GPS_ALL_PORT_INFO,
                          IOCTL_GET_GPS_ALL_PORT_INFO, pii,
                          p_ri->n_com_ports * sizeof( pii[0] ),
                          _pcps_ddev_has_receiver_info( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_get_gps_all_port_info



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_set_gps_port_settings_idx( MBG_DEV_HANDLE dh,
                                                          const PORT_SETTINGS_IDX *p )
{
  _mbgdevio_vars();
  _mbgdevio_write_gps_var_chk( dh, PC_GPS_PORT_SETTINGS_IDX,
                               IOCTL_SET_GPS_PORT_SETTINGS_IDX, p,
                               _pcps_ddev_has_receiver_info( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_set_gps_port_settings_idx



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_setup_receiver_info( MBG_DEV_HANDLE dh,
                                                    const PCPS_DEV *pdev,
                                                    RECEIVER_INFO *p )
{
  // If the clock supports the receiver_info structure then
  // read it from the clock, otherwise set up some default
  // values depending on the clock type.
  if ( _pcps_has_receiver_info( pdev ) )
    return mbg_get_gps_receiver_info( dh, p );


  if ( _pcps_is_gps( pdev ) )
  { // keep braces due to the macro call!
    _setup_default_receiver_info_gps( p );
  }
  else
  { // keep braces due to the macro call!
    _setup_default_receiver_info_dcf( p, pdev );
  }

  // Make sure this program supports at least as many ports as
  // the current clock device.
  if ( p->n_com_ports > MAX_PARM_PORT )
    return -1; //##++

  // Make sure this program supports at least as many string types
  // as the current clock device.
  if ( p->n_str_type > MAX_PARM_STR_TYPE )
    return -2; //##++


  return PCPS_SUCCESS;

}  // mbg_setup_receiver_info



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_serial_settings( MBG_DEV_HANDLE dh,
                                                    const PCPS_DEV *pdev,
                                                    RECEIVER_PORT_CFG *pcfg,
                                                    const RECEIVER_INFO *p_ri )
{
  int rc;
  int i;

  memset( pcfg, 0, sizeof( *pcfg ) );

  if ( _pcps_has_receiver_info( pdev ) )
  {
    rc = mbg_get_gps_all_port_info( dh, pcfg->pii, p_ri );
    if ( rc != PCPS_SUCCESS )
      goto error;

    rc = mbg_get_gps_all_str_type_info( dh, pcfg->stii, p_ri );
    if ( rc != PCPS_SUCCESS )
      goto error;
  }
  else
  {
    if ( _pcps_is_gps( pdev ) )
    {
      rc = mbg_get_gps_port_parm( dh, &pcfg->tmp_pp );
      if ( rc != PCPS_SUCCESS )
        goto error;

      for ( i = 0; i < p_ri->n_com_ports; i++ )
      {
        PORT_INFO_IDX *p_pii = &pcfg->pii[i];
        PORT_INFO *p_pi = &p_pii->port_info;

        p_pii->idx = i;
        port_settings_from_port_parm( &p_pi->port_settings,
                                      i, &pcfg->tmp_pp, 1 );

        p_pi->supp_baud_rates = DEFAULT_GPS_BAUD_RATES_C166;
        p_pi->supp_framings = DEFAULT_GPS_FRAMINGS_C166;
        p_pi->supp_str_types = DEFAULT_SUPP_STR_TYPES_GPS;
      }
    }
    else
    {
      PCPS_SERIAL ser_code;

      rc = mbg_get_serial( dh, &ser_code );
      if ( rc != PCPS_SUCCESS )
        goto error;


      port_info_from_pcps_serial( pcfg->pii, ser_code,
                                  _pcps_has_serial_hs( pdev ) ?
                                    DEFAULT_BAUD_RATES_DCF_HS :
                                    DEFAULT_BAUD_RATES_DCF
                               );
    }

    for ( i = 0; i < p_ri->n_str_type; i++ )
    {
      STR_TYPE_INFO_IDX *stip = &pcfg->stii[i];
      stip->idx = i;
      stip->str_type_info = default_str_type_info[i];
    }
  }

  return PCPS_SUCCESS;


error:
  return rc;

}  // mbg_get_serial_settings



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_save_serial_settings( MBG_DEV_HANDLE dh,
                                                     const PCPS_DEV *pdev,
                                                     RECEIVER_PORT_CFG *pcfg,
                                                     int port_num )
{
  int rc;

  if ( _pcps_has_receiver_info( pdev ) )
  {
    PORT_SETTINGS_IDX psi;
    psi.idx = port_num;
    psi.port_settings = pcfg->pii[port_num].port_info.port_settings;

    rc = mbg_set_gps_port_settings_idx( dh, &psi );
  }
  else
  {
    if ( _pcps_is_gps( pdev ) )
    {
      port_parm_from_port_settings( &pcfg->tmp_pp, port_num,
                          &pcfg->pii[port_num].port_info.port_settings, 1 );

      rc = mbg_set_gps_port_parm( dh, &pcfg->tmp_pp );
    }
    else
    {
      PCPS_SERIAL ser_code;

      pcps_serial_from_port_info( &ser_code, pcfg->pii );

      rc = mbg_set_serial( dh, &ser_code );
    }
  }

  return rc;

}  // mbg_save_serial_settings



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_asic_version( MBG_DEV_HANDLE dh, PCPS_ASIC_VERSION *p )
{

  #if defined( _MBGIOCTL_H )
    _mbgdevio_vars();
    _mbgdevio_read_var( dh, -1, IOCTL_GET_PCPS_ASIC_VERSION, p );
    return _mbgdevio_ret_val;
  #else
    if ( !_pcps_ddev_is_pci_asic( dh ) )
      return -1;

    *p = inpd( _pcps_ddev_port_base( dh, 0 )
             + offsetof( PCI_ASIC, features ) );

    return PCPS_SUCCESS;
  #endif

}  // mbg_get_asic_version



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_time_cycles( MBG_DEV_HANDLE dh, PCPS_TIME_CYCLES *p )
{
  _mbgdevio_vars();
  _mbgdevio_read_var( dh, PCPS_GIVE_TIME, IOCTL_GET_PCPS_TIME_CYCLES, p );
  #if !defined( _MBGIOCTL_H )
    // only if not using IOCTLs
    // for PCPS_TIME, read stamp AFTER the call
    p->cycles = 0;  //##++
  #endif
  return _mbgdevio_ret_val;

}  // mbg_get_time_cycles



/*HDR*/
_MBG_API_ATTR int _MBG_API mbg_get_hr_time_cycles( MBG_DEV_HANDLE dh,
                                                   PCPS_HR_TIME_CYCLES *p )
{
  _mbgdevio_vars();
  #if !defined( _MBGIOCTL_H )
    // only if not using IOCTLs
    // for PCPS_TIME, read stamp BEFORE the call
    p->cycles = 0;  //##++
  #endif
  _mbgdevio_read_var_chk( dh, PCPS_GIVE_HR_TIME,
                          IOCTL_GET_PCPS_HR_TIME_CYCLES,
                          p, _pcps_ddev_has_hr_time( dh ) );
  return _mbgdevio_ret_val;

}  // mbg_get_hr_time_cycles



#if defined( MBG_TGT_WIN32 )

BOOL APIENTRY DllMain( HANDLE hModule,
                       DWORD  ul_reason_for_call,
                       LPVOID lpReserved )
{
  return TRUE;
}

#endif  // defined( MBG_TGT_WIN32 )

