
/**************************************************************************
 *
 *  $Id: gpsutils.c 1.5 2003/02/04 09:20:04Z MARTIN TEST $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Utility functions useful with GPS data.
 *
 * -----------------------------------------------------------------------
 *  $Log: gpsutils.c $
 *  Revision 1.5  2003/02/04 09:20:04Z  MARTIN
 *  New functions sprint_alt(), sprint_fixed_freq().
 *  Revision 1.4  2003/01/31 13:45:19  MARTIN
 *  sprint_pos_geo() returns N/A if position not valid.
 *  Revision 1.3  2002/12/12 16:07:04  martin
 *  New functions swap_pos_doubles(), sprint_dms(),
 *  and sprint_pos_geo().
 *  Revision 1.2  2001/02/05 09:39:12Z  MARTIN
 *  New file header.
 *  Change include file name to lower case.
 *  Source code cleanup.
 *
 **************************************************************************/

#define _GPSUTILS
 #include <gpsutils.h>
#undef _GPSUTILS

#include <stdio.h>
#include <string.h>
#include <math.h>


#define _eos( _s )  ( &(_s)[strlen( _s )] )


/*HDR*/
void swap_double( double *d )
{
  ushort *wp1;
  ushort *wp2;
  ushort w;
  int i;

  wp1 = (ushort *) d;
  wp2 = ( (ushort *) d ) + 3;

  for ( i = 0; i < 2; i++ )
  {
    w = *wp1;
    *wp1 = *wp2;
    *wp2 = w;
    wp1++;
    wp2--;
  }

}  /* swap_double */



/*HDR*/
void swap_eph_doubles( EPH *ephp )
{
  swap_double( &ephp->sqrt_A );
  swap_double( &ephp->e );
  swap_double( &ephp->M0 );
  swap_double( &ephp->omega );
  swap_double( &ephp->i0 );
  swap_double( &ephp->OMEGA0 );
  swap_double( &ephp->OMEGADOT );

  swap_double( &ephp->deltan );
  swap_double( &ephp->idot );

  swap_double( &ephp->crc );
  swap_double( &ephp->crs );
  swap_double( &ephp->cuc );
  swap_double( &ephp->cus );
  swap_double( &ephp->cic );
  swap_double( &ephp->cis );

  swap_double( &ephp->af0 );
  swap_double( &ephp->af1 );
  swap_double( &ephp->af2 );

  swap_double( &ephp->tgd );

}  /* swap_eph_doubles */



/*HDR*/
void swap_alm_doubles( ALM *almp )
{
  swap_double( &almp->sqrt_A );
  swap_double( &almp->e );
  swap_double( &almp->deltai );
  swap_double( &almp->OMEGA0 );
  swap_double( &almp->OMEGADOT );
  swap_double( &almp->omega );
  swap_double( &almp->M0 );
  swap_double( &almp->af0 );
  swap_double( &almp->af1 );

}  /* swap_alm_doubles */



/*HDR*/
void swap_utc_doubles( UTC *utcp )
{
  swap_double( &utcp->A0 );
  swap_double( &utcp->A1 );

}  /* swap_utc_doubles */



/*HDR*/
void swap_iono_doubles( IONO *ionop )
{
  swap_double( &ionop->alpha_0 );
  swap_double( &ionop->alpha_1 );
  swap_double( &ionop->alpha_2 );
  swap_double( &ionop->alpha_3 );

  swap_double( &ionop->beta_0 );
  swap_double( &ionop->beta_1 );
  swap_double( &ionop->beta_2 );
  swap_double( &ionop->beta_3 );

}  /* swap_iono_doubles */



/*HDR*/
void swap_pos_doubles( POS *posp )
{
  int i;

  for ( i = 0; i < N_XYZ; i++ )
    swap_double( &posp->xyz[i] );

  for ( i = 0; i < N_LLA; i++ )
    swap_double( &posp->lla[i] );

  swap_double( &posp->longitude.sec );
  swap_double( &posp->latitude.sec );

}  /* swap_pos_doubles */



/*HDR*/
void sprint_dms( char *s, DMS *pdms, int prec )
{
  sprintf( s, "%c %i" S_DEGREE "%02i'%02.*f\"",
           pdms->prefix,
           pdms->deg,
           pdms->min,
           prec,
           pdms->sec
         );

}  /* sprint_dms */



/*HDR*/
void sprint_alt( char *s, double alt )
{
  sprintf( s, "%.0fm", alt );

}  /* sprint_dms */



/*HDR*/
void sprint_pos_geo( char *s, POS *ppos, const char *sep, int prec )
{
  if ( ppos->lla[LON] && ppos->lla[LAT] && ppos->lla[ALT] )
  {
    sprint_dms( s, &ppos->latitude, prec );
    strcat( s, sep );
    sprint_dms( _eos( s ), &ppos->longitude, prec );
    strcat( s, sep );
    sprint_alt( _eos( s ), ppos->lla[ALT] );
  }
  else
    strcpy( s, "N/A" );

}  /* sprint_pos_geo */



/*HDR*/
void sprint_fixed_freq( char *s, FIXED_FREQ_INFO *p_ff )
{
  double freq;
  int range;
  ushort unit;
  ushort format;

  // before re-calculating frequency range is the base 10 exponent
  // to the frequency value which is represented in khz
  // after calculating range from real frequencyl, range is represented
  // as follows
  //    range  display  format  divisor    format index calculation
  //     -3    100mHz   1.000  [/1e-3]   -3 % 3 = -3 + 3 = 0 % 3 = 0
  //     -2    100mHz   10.00  [/1e-3]   -2 % 3 = -2 + 3 = 1 % 3 = 1
  //     -1    100mHz   100.0  [/1e-3]   -1 % 3 = -1 + 3 = 2 % 3 = 2
  //      0    1Hz      1.000  [/1e0]     0 % 3 =  0 + 3 = 3 % 3 = 0
  //      1    10Hz     10.00  [/1e0]     1 % 3 =  1 + 3 = 1 % 3 = 1
  //      2    100Hz    100.0  [/1e0]     2 % 3 =  2 + 3 = 2 % 3 = 2
  //      3    1kHz     1.000  [/1e3]     3 % 3 =  0 + 3 = 3 % 3 = 0
  //      4    10kHz    10.00  [/1e3]     4 % 3 =  1 + 3 = 4 % 3 = 1
  //      5    100kHz   100.0  [/1e3]     5 % 3 =  2 + 3 = 5 % 3 = 2
  //      6    1MHz     1.000  [/1e6]     6 % 3 =  0 + 3 = 3 % 3 = 0
  //      7    10MHz    10.00  [/1e6]     7 % 3 =  1 + 3 = 4 % 3 = 1
  //      8    100MHz   100.0  [/1e6]     8 % 3 =  2 + 3 = 5 % 3 = 2

  // format string for fp output
  static const char *fmt_str[] =
  {
    "%4.3lf%s",
    "%4.2lf%s",
    "%4.1lf%s",
  };

  // unit index and divisor are calculated as follows
  // range  unit        index calculation            divisor calculation
  // -3      mHz   ( int )( ( -3 + 3 ) / 3 ) =  0
  // -2      mHz   ( int )( ( -2 + 3 ) / 3 ) =  0
  // -1      mHz   ( int )( ( -1 + 3 ) / 3 ) =  0    / 10e-3 = 10e( 3 * 0 - 3 )
  //  0      Hz    ( int )( (  0 + 3 ) / 3 ) =  1
  //  1      Hz    ( int )( (  1 + 3 ) / 3 ) =  1
  //  2      Hz    ( int )( (  2 + 3 ) / 3 ) =  1    / 1e0   = 10e( 3 * 1 - 3 )
  //  3      kHz   ( int )( (  3 + 3 ) / 3 ) =  2
  //  4      kHz   ( int )( (  4 + 3 ) / 3 ) =  2
  //  5      kHz   ( int )( (  5 + 3 ) / 3 ) =  2    / 10e3  = 10e( 3 * 2 - 3 )
  //  6      MHz   ( int )( (  6 + 3 ) / 3 ) =  3
  //  7      MHz   ( int )( (  7 + 3 ) / 3 ) =  3
  //  8      MHz   ( int )( (  8 + 3 ) / 3 ) =  3    / 10e6   =10e( 3 * 3 - 3 )

  // unit string
  static const char *unit_str[] =
  {
    "mHz",
    "Hz",
    "kHz",
    "MHz"
  };


  // calculate frequency in Hz
  freq = ( double ) p_ff->khz_val * pow( 10, ( p_ff->range + 3 ) );

  // calculate decimal exponent
  range = ( ( int ) log10( freq ) );

  // check whether range is in the allowed range
  // if so display frequency in broken format
  if ( ( range >= -3 ) && ( range <= 8 ) )
  {
    // calculate format index ( see above )
    format = ( ( ( range % 3 ) + 3 ) % 3 );

    // calculate unit index
    unit =  ( ushort )( range + 3 ) / 3;

    // calculate display value
    freq = freq / pow( 10, ( ( 3 * unit ) - 3 ) );
    sprintf( s, fmt_str[format], freq, unit_str[unit] );
    return;
  }
  else
  {
    // out of range display fequency in Hz
    sprintf( s, "%lfHz", freq );
  }

}  /* sprint_fixed_freq */


