
/**************************************************************************
 *
 *  $Id: gpsdefs.h 1.33 2004/11/09 12:39:59Z martin TEST $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    General definitions to be used with Meinberg GPS clocks.
 *
 * -----------------------------------------------------------------------
 *  $Log: gpsdefs.h $
 *  Revision 1.33  2004/11/09 12:39:59Z  martin
 *  Redefined interface data types using C99 fixed-size definitions.
 *  Added model code and name for TCR167PCI.
 *  New type GPS_CMD.
 *  Defined type BVAR_STAT and associated flags.
 *  Revision 1.32  2004/09/20 12:46:25  andre
 *  Added structures and definitions for SCU board.
 *  Revision 1.31  2004/07/08 08:30:36Z  martin
 *  Added feature GPS_FEAT_RCV_TIMEOUT.
 *  Revision 1.30  2004/06/21 13:38:42  martin
 *  New flag MBG_OPT_BIT_EMU_SYNC/MBG_OPT_BIT_EMU_SYNC
 *  lets the receicer emulate/pretend to be always synchronized.
 *  Revision 1.30  2004/06/21 13:35:46Z  martin
 *  Revision 1.29  2004/06/16 12:47:53Z  martin
 *  Moved OPT_SETTINGS related definitions from pcpsdefs.h
 *  here and renamed symbols from PCPS_.. to to MBG_...
 *  Revision 1.28  2004/03/26 10:37:00Z  martin
 *  Added definitions to support multiple ref sources.
 *  Added definitions OSC_DAC_RANGE, OSC_DAC_BIAS.
 *  Revision 1.27  2004/03/08 14:06:45Z  martin
 *  New model code and name for GPS169PCI.
 *  Existing feature GPS_FEAT_IRIG has been 
 *  renamed to GPS_FEAT_IRIG_TX.
 *  Added feature GPS_FEAT_IRIG_RX.
 *  Added IPv4 LAN interface feature flags.
 *  Renamed IFLAGS_IGNORE_TFOM to IFLAGS_DISABLE_TFOM.
 *  Revision 1.26  2003/12/05 12:28:20Z  martin
 *  Added some codes used with IRIG cfg.
 *  Revision 1.25  2003/10/29 16:18:14Z  martin
 *  Added 7N2 to DEFAULT_GPS_FRAMINGS_GP2021.
 *  Revision 1.24  2003/09/30 08:49:48Z  martin
 *  New flag TM_LS_ANN_NEG which is set in addition to
 *  TM_LS_ANN if next leap second is negative.
 *  Revision 1.23  2003/08/26 14:32:33Z  martin
 *  Added some initializers for commonly used
 *  TZDL configurations.
 *  Revision 1.22  2003/04/25 10:18:11  martin
 *  Fixed typo inside an IRIG name string initializer.
 *  Revision 1.21  2003/04/15 09:18:48  martin
 *  New typedef ANT_CABLE_LEN.
 *  Revision 1.20  2003/04/03 11:03:44Z  martin
 *  Extended definitions for IRIG support.
 *  Revision 1.19  2003/01/31 13:38:20  MARTIN
 *  Modified type of RECEIVER_INFO.fixed_freq field.
 *  Revision 1.18  2002/10/28 09:24:07  MARTIN
 *  Added/renamed some POUT related symbols.
 *  Revision 1.17  2002/09/05 10:58:39  MARTIN
 *  Renamed some symbols related to programmable outputs.
 *  Revision 1.16  2002/08/29 08:04:47  martin
 *  Renamed structure POUT_PROG to POUT_SETTINGS.
 *  New structures POUT_SETTINGS_IDX, POUT_INFO,
 *  POUT_INFO_IDX and associated definitions.
 *  Updated some comments.
 *  Revision 1.15  2002/07/17 07:39:39Z  Andre
 *  comma added in definition DEFAULT_GPS_OSC_NAMES
 *  Revision 1.14  2002/06/27 12:17:29Z  MARTIN
 *  Added new oscillator code TCXO_MQ.
 *  Added initializer for oscillator names.
 *  Added initializer for oscillator list ordered by quality.
 *  Revision 1.13  2002/05/08 08:16:03  MARTIN
 *  Added GPS_OSC_CFG_SUPP for RECEIVER_INFO.flags.
 *  Fixed some comments.
 *  Revision 1.12  2002/03/14 13:45:56  MARTIN
 *  Changed type CSUM from short to ushort.
 *  Revision 1.11  2002/03/01 12:29:30  Andre
 *  Added GPS_MODEL_GPS161 and GPS_MODEL_NAME_GPS161.
 *  Revision 1.10  2002/02/25 08:02:33Z  MARTIN
 *  Added array of chars to union IDENT.
 *  Revision 1.9  2002/01/29 15:21:46  MARTIN
 *  Added new field "reserved" to struct SW_REV to fix C166 data
 *  alignment/structure size. Converted structure IDENT to a union.
 *  The changes above should not affect existing monitoring programs.
 *  New status flag TM_ANT_SHORT.
 *  New structure RECEIVER_INFO and associated definitions to
 *  enhance control from monitoring programs.
 *  New structures PORT_INFO, STR_TYPE_INFO, and associated
 *  definitions to simplify and unify configuration from external programs.
 *  New structures IRIG_INFO and POUT_PROG_IDX to configure an
 *  optional IRIG interface and programmable pulse outputs.
 *  Modified some comments.
 *  Revision 1.8  2001/03/30 11:44:11  MARTIN
 *  Control alignment of structures from new file use_pack.h.
 *  Defined initializers with valid baud rate and framing parameters.
 *  Modified some comments.
 *  Revision 1.7  2001/03/01 08:09:22  MARTIN
 *  Modified preprocessor syntax.
 *  Revision 1.6  2000/07/21 14:04:33  MARTIN
 *  Added som #if directives to protect structures against being multiply
 *  defined.
 *  Modified some comments.
 *  Comments using characters for +/- and degree now include ASCII
 *  characters only.
 *
 **************************************************************************/

#ifndef _GPSDEFS_H
#define _GPSDEFS_H


/* Other headers to be included */

#include <words.h>
#include <use_pack.h>


#if defined( _USE_PACK )      // set byte alignment
  #pragma pack( 1 )
#endif


/* Start of header body */

#define MIN_SVNO         1                  /* min. SV number */
#define MAX_SVNO        32                  /* max. SV number */
#define N_SVNO ( MAX_SVNO - MIN_SVNO + 1)   /* number of possibly active SVs */


#define GPS_ID_STR_LEN      16
#define GPS_ID_STR_SIZE     ( GPS_ID_STR_LEN + 1 )

#define GPS_EPLD_STR_LEN    8
#define GPS_EPLD_STR_SIZE   ( GPS_EPLD_STR_LEN + 1 )


#define DEFAULT_GPS_TICKS_PER_SEC   10000000L  /* system time base */

#if !defined( GPS_TICKS_PER_SEC )
  /*
   * The actual ticks per seconds may vary for different
   * GPS receiver models. If this is the case, the receiver
   * model support the RECEIVER_INFO structure which contains
   * the actual value.
   */
  #define GPS_TICKS_PER_SEC   DEFAULT_GPS_TICKS_PER_SEC
#endif


typedef int16_t SVNO;     /* the number of a SV */
typedef uint16_t HEALTH;  /* a SV's health code */
typedef uint16_t CFG;     /* a SV's configuration code */
typedef uint16_t IOD;     /* Issue-Of-Data code */


/* the type of various checksums */

#ifndef _CSUM_DEFINED
  typedef uint16_t CSUM;
  #define _CSUM_DEFINED
#endif


// The type which is used to pass a cmd code via serial port, or bus.
// The cmd codes are defined in gpsserio.h and pcpsdefs.h.
typedef uint16_t GPS_CMD;


/* a struct used to hold the software revision information */

typedef struct
{
  uint16_t code;               /* e.g. 0x0120 means rev. 1.20 */
  char name[GPS_ID_STR_SIZE];  /* used to identify customized versions */
  uint8_t reserved;            /* yield even structure size */
} SW_REV;



typedef uint16_t BVAR_STAT; /* holds status of battery buffered vars */

// The bits defined below are set in BVAR_STAT if the corresponding
// parameters are NOT valid and complete:

enum
{
  BVAR_BIT_CFGH_INVALID,
  BVAR_BIT_ALM_NOT_COMPLETE,
  BVAR_BIT_UTC_INVALID,
  BVAR_BIT_IONO_INVALID,
  BVAR_BIT_RCVR_POS_INVALID,
  N_BVAR_BIT   // number of defined bits
};

#define BVAR_CFGH_INVALID      ( 1UL << BVAR_BIT_CFGH_INVALID )
#define BVAR_ALM_NOT_COMPLETE  ( 1UL << BVAR_BIT_ALM_NOT_COMPLETE )
#define BVAR_UTC_INVALID       ( 1UL << BVAR_BIT_UTC_INVALID )
#define BVAR_IONO_INVALID      ( 1UL << BVAR_BIT_IONO_INVALID )
#define BVAR_RCVR_POS_INVALID  ( 1UL << BVAR_BIT_RCVR_POS_INVALID )

// bit mask for all defined bits
#define BVAR_MASK  ( ( 1UL << N_BVAR_BIT ) - 1 )



/* a struct used to hold a fixed frequency value */
/* frequ[kHz] = khz_val * 10^range */

typedef struct
{
  uint16_t khz_val;     /* the base frequency in [kHz] */
  int16_t range;        /* an optional base 10 exponent */
} FIXED_FREQ_INFO;



/*
 * The following code defines features and properties
 * of the various GPS receivers. Older GPS receivers
 * may require a recent firmvare version to support
 * this, or may not support this at all.
 */

/*
 * The structure is ordered in a way that all fields
 * except chars or arrays of chars are word-aligned.
 */
typedef struct
{
  uint16_t model_code;               /* identifier for receiver model */
  SW_REV sw_rev;                     /* software revision and ID */
  char model_name[GPS_ID_STR_SIZE];  /* ASCIIZ, name of receiver model */
  char sernum[GPS_ID_STR_SIZE];      /* ASCIIZ, serial number */
  char epld_name[GPS_EPLD_STR_SIZE]; /* ASCIIZ, file name of EPLD image */
  uint8_t n_channels;       /* number of sats to be tracked simultaneously */
  uint32_t ticks_per_sec;   /* resolution of fractions of seconds */
  uint32_t features;        /* optional features, see below */
  FIXED_FREQ_INFO fixed_freq; /* optional non-standard fixed frequency */
  uint8_t osc_type;         /* type of installed oscillator, see below */
  uint8_t osc_flags;        /* oscillator flags, see below */
  uint8_t n_ucaps;          /* number of user time capture inputs */
  uint8_t n_com_ports;      /* number of on-board serial ports */
  uint8_t n_str_type;       /* max num of string types supported by any port */
  uint8_t n_prg_out;        /* number of programmable pulse outputs */
  uint16_t flags;           /* additional information, see below */
} RECEIVER_INFO;


/*
 * Valid codes for RECEIVER_INFO.model_code:
 */
enum
{
  GPS_MODEL_UNKNOWN,
  GPS_MODEL_GPS166,
  GPS_MODEL_GPS167,
  GPS_MODEL_GPS167SV,
  GPS_MODEL_GPS167PC,
  GPS_MODEL_GPS167PCI,
  GPS_MODEL_GPS163,
  GPS_MODEL_GPS168PCI,
  GPS_MODEL_GPS161,
  GPS_MODEL_GPS169PCI,
  GPS_MODEL_TCR167PCI,
  N_GPS_MODEL
};


/*
 * String initializers for each of the GPS
 * receiver models enum'ed above:
 */
#define GPS_MODEL_NAME_UNKNOWN   "(unknown)"
#define GPS_MODEL_NAME_GPS166    "GPS166"
#define GPS_MODEL_NAME_GPS167    "GPS167"
#define GPS_MODEL_NAME_GPS167SV  "GPS167SV"
#define GPS_MODEL_NAME_GPS167PC  "GPS167PC"
#define GPS_MODEL_NAME_GPS167PCI "GPS167PCI"
#define GPS_MODEL_NAME_GPS163    "GPS163"
#define GPS_MODEL_NAME_GPS168PCI "GPS168PCI"
#define GPS_MODEL_NAME_GPS161    "GPS161"
#define GPS_MODEL_NAME_GPS169PCI "GPS169PCI"
#define GPS_MODEL_NAME_TCR167PCI "TCR167PCI"


/*
 * The definition below can be used to initialize
 * an array of N_GPS_MODEL type name strings.
 * Including the trailing 0, each name must not
 * exceed GPS_ID_STR_SIZE chars.
 */
#define DEFAULT_GPS_MODEL_NAMES \
{                               \
  GPS_MODEL_NAME_UNKNOWN,       \
  GPS_MODEL_NAME_GPS166,        \
  GPS_MODEL_NAME_GPS167,        \
  GPS_MODEL_NAME_GPS167SV,      \
  GPS_MODEL_NAME_GPS167PC,      \
  GPS_MODEL_NAME_GPS167PCI,     \
  GPS_MODEL_NAME_GPS163,        \
  GPS_MODEL_NAME_GPS168PCI,     \
  GPS_MODEL_NAME_GPS161,        \
  GPS_MODEL_NAME_GPS169PCI,     \
  GPS_MODEL_NAME_TCR167PCI      \
}


/*
 * The classification codes for oscillators below
 * are used with RECEIVER_INFO.osc_type. New codes
 * must be appended to the enumeration, so the sequence
 * of codes does NOT reflect the order of quality:
 */
enum
{
  GPS_OSC_UNKNOWN,
  GPS_OSC_TCXO_LQ,
  GPS_OSC_TCXO_HQ,
  GPS_OSC_OCXO_LQ,
  GPS_OSC_OCXO_MQ,
  GPS_OSC_OCXO_HQ,
  GPS_OSC_OCXO_XHQ,
  GPS_OSC_RUBIDIUM,
  GPS_OSC_TCXO_MQ,
  N_GPS_OSC
};


/*
 * The sequence and number of oscillator names
 * listed below must correspond to the enumeration
 * above:
 */
#define DEFAULT_GPS_OSC_NAMES \
{                             \
  "- unknown -",              \
  "TCXO LQ",                  \
  "TCXO HQ",                  \
  "OCXO LQ",                  \
  "OCXO MQ",                  \
  "OCXO HQ",                  \
  "OCXO XHQ",                 \
  "RUBIDIUM",                 \
  "TCXO MQ"                   \
}


/*
 * The initializer below can be used to initialize
 * an array (e.g. "int osc_quality_idx[N_GPS_OSC]")
 * which allows to display the oscillator types
 * ordered by quality:
 */
#define DEFAULT_GPS_OSC_QUALITY_IDX \
{                                   \
  GPS_OSC_UNKNOWN,                  \
  GPS_OSC_TCXO_LQ,                  \
  GPS_OSC_TCXO_MQ,                  \
  GPS_OSC_TCXO_HQ,                  \
  GPS_OSC_OCXO_LQ,                  \
  GPS_OSC_OCXO_MQ,                  \
  GPS_OSC_OCXO_HQ,                  \
  GPS_OSC_OCXO_XHQ,                 \
  GPS_OSC_RUBIDIUM                  \
}



/*
 * Codes to be used with RECEIVER_INFO.osc_flags
 * are not yet used/required, so they are reserved
 * for future use.
 */


/*
 * The codes below enumerate some features which may be
 * supported by a given clock, or not.
 */
enum
{
  GPS_FEAT_PPS,         /* has pulse per second output */
  GPS_FEAT_PPM,         /* has pulse per minute output */
  GPS_FEAT_SYNTH,       /* has programmable synthesizer output */
  GPS_FEAT_DCFMARKS,    /* has DCF77 compatible time mark output */
  GPS_FEAT_IRIG_TX,     /* has on-board IRIG output */
  GPS_FEAT_IRIG_RX,     /* has on-board IRIG input */
  GPS_FEAT_LAN_IP4,     /* has LAN IPv4 interface */
  GPS_FEAT_MULTI_REF,   /* has multiple input sources with priorities */
  GPS_FEAT_RCV_TIMEOUT, /* timeout after GPS reception has stopped */
  N_GPS_FEATURE         /* the number of valid features */
};


#define DEFAULT_GPS_FEATURE_NAMES \
{                                 \
  "Pulse Per Second",             \
  "Pulse Per Minute",             \
  "Programmable Synth.",          \
  "DCF77 Time Marks",             \
  "IRIG Out",                     \
  "IRIG In",                      \
  "IPv4 LAN Interface",           \
  "Multiple Ref. Sources",        \
  "GPS receive timeout"           \
}


/*
 * Bit masks used with RECEIVER_INFO.features
 * (others are reserved):
 */
#define GPS_HAS_PPS         ( 1UL << GPS_FEAT_PPS )
#define GPS_HAS_PPM         ( 1UL << GPS_FEAT_PPM )
#define GPS_HAS_SYNTH       ( 1UL << GPS_FEAT_SYNTH )
#define GPS_HAS_DCFMARKS    ( 1UL << GPS_FEAT_DCFMARKS )
#define GPS_HAS_IRIG_TX     ( 1UL << GPS_FEAT_IRIG_TX )
#define GPS_HAS_IRIG_RX     ( 1UL << GPS_FEAT_IRIG_RX )
#define GPS_HAS_LAN_IP4     ( 1UL << GPS_FEAT_LAN_IP4 )
#define GPS_HAS_MULTI_REF   ( 1UL << GPS_FEAT_MULTI_REF )
#define GPS_HAS_RCV_TIMEOUT ( 1UL << GPS_FEAT_RCV_TIMEOUT )

/*
 * The features below are supported by default by older
 * C166 based GPS receivers:
 */
#define DEFAULT_GPS_FEATURES_C166 \
{                                 \
  GPS_HAS_PPS |                   \
  GPS_HAS_PPM |                   \
  GPS_HAS_SYNTH |                 \
  GPS_HAS_DCFMARKS                \
}


/*
 * Codes to be used with RECEIVER_INFO.flags:
 */
#define GPS_OSC_CFG_SUPP    0x01  // GPS_OSC_CFG supported



/* Date and time referred to the linear time scale defined by GPS. */
/* GPS time is defined by the number of weeks since midnight from */
/* January 5, 1980 to January 6, 1980 plus the number of seconds of */
/* the current week plus fractions of a second. GPS time differs from */
/* UTC because UTC is corrected with leap seconds while GPS time scale */
/* is continuous. */

typedef struct
{
  uint16_t wn;     /* the week number since GPS has been installed */
  uint32_t sec;    /* the second of that week */
  uint32_t tick;   /* fractions of a second; scale: 1/GPS_TICKS_PER_SEC */
} T_GPS;



/* Local date and time computed from GPS time. The current number */
/* of leap seconds have to be added to get UTC from GPS time. */
/* Additional corrections could have been made according to the */
/* time zone/daylight saving parameters (TZDL, see below) defined */
/* by the user. The status field can be checked to see which corrections */
/* have been applied. */

typedef struct
{
  int16_t year;           /* 0..9999 */
  int8_t month;           /* 1..12 */
  int8_t mday;            /* 1..31 */
  int16_t yday;           /* 1..366 */
  int8_t wday;            /* 0..6 == Sun..Sat */
  int8_t hour;            /* 0..23 */
  int8_t min;             /* 0..59 */
  int8_t sec;             /* 0..59 */
  int32_t frac;           /* fractions of a second; scale: 1/GPS_TICKS_PER_SEC*/
  int32_t offs_from_utc;  /* local time's offset from UTC */
  uint16_t status;        /* flags */
} TM_GPS;

/* status flags used with conversion from GPS time to local time */

#define TM_UTC        0x01   /* UTC correction has been made */
#define TM_LOCAL      0x02   /* UTC has been converted to local time */
#define TM_DL_ANN     0x04   /* state of daylight saving is going to change */
#define TM_DL_ENB     0x08   /* daylight saving is enabled */
#define TM_LS_ANN     0x10   /* leap second will be inserted */
#define TM_LS_ENB     0x20   /* current second is leap second */
#define TM_LS_ANN_NEG 0x40   /* set in addition to TM_LS_ANN if leap sec negative */



/* the status flags below were defined starting with GPS166 v1.32 */

#define TM_ANT_SHORT    0x0400  /* antenna cable short circuited */
#define TM_NO_WARM      0x0800  /* OCXO has not warmed up */
#define TM_ANT_DISCONN  0x1000  /* antenna currently disconnected */
#define TM_SYN_FLAG     0x2000  /* TIME_SYN output is low */
#define TM_NO_SYNC      0x4000  /* not sync'ed after reset */
#define TM_NO_POS       0x8000  /* position not computed after reset, */
                                /*   LOCK LED off */

/* a struct used to transmit information on date and time */

typedef struct
{
  int16_t channel;      /* -1: the current time; 0, 1: capture 0, 1 */
  T_GPS t;              /* time in GPS format */
  TM_GPS tm;            /* that time converted to local time */
} TTM;



/* Two types of variables used to store a position. Type XYZ is */
/* used with a position in earth centered, earth fixed (ECEF) */
/* coordinates whereas type LLA holds such a position converted */
/* to geographic coordinates as defined by WGS84 (World Geodetic */
/* System from 1984). */

#ifndef _XYZ_DEFINED
  /* sequence and number of components of a cartesian position */
  enum { XP, YP, ZP, N_XYZ };

  /* a type of array holding a cartesian position */
  typedef double XYZ[N_XYZ];      /* values are in [m] */

  #define _XYZ_DEFINED
#endif


#ifndef _LLA_DEFINED
  /* sequence and number of components of a geographic position */
  enum { LAT, LON, ALT, N_LLA };  /* latitude, longitude, altitude */

  /* a type of array holding a geographic position */
  typedef double LLA[N_LLA];      /* lon, lat in [rad], alt in [m] */

  #define _LLA_DEFINED
#endif



/* Synthesizer parameters. Synthesizer frequency is expressed as a */
/* four digit decimal number (freq) to be multiplied by 0.1 Hz and an */
/* base 10 exponent (range). If the effective frequency is less than */
/* 10 kHz its phase is synchronized corresponding to the variable phase. */
/* Phase may be in a range from -360 deg to +360 deg with a resolution */
/* of 0.1 deg, so the resulting numbers to be stored are in a range of */
/* -3600 to +3600. */

/* Example: */
/* Assume the value of freq is 2345 (decimal) and the value of phase is 900. */
/* If range == 0 the effective frequency is 234.5 Hz with a phase of +90 deg. */
/* If range == 1 the synthesizer will generate a 2345 Hz output frequency */
/* and so on. */

/* Limitations: */
/* If freq == 0 the synthesizer is disabled. If range == 0 the least */
/* significant digit of freq is limited to 0, 3, 5 or 6. The resulting */
/* frequency is shown in the examples below: */
/*     freq == 1230  -->  123.0 Hz */
/*     freq == 1233  -->  123 1/3 Hz (real 1/3 Hz, NOT 123.3 Hz) */
/*     freq == 1235  -->  123.5 Hz */
/*     freq == 1236  -->  123 2/3 Hz (real 2/3 Hz, NOT 123.6 Hz) */

/* If range == MAX_RANGE the value of freq must not exceed 1200, so the */
/* output frequency is limited to 12 MHz. */

/* Phase will be ignored if the resulting frequency is greater or equal */
/* to 10 kHz. */

#define MAX_SYNTH_FREQ   1200    /* if range == MAX_SYNTH_RANGE */
#define MIN_SYNTH_RANGE     0
#define MAX_SYNTH_RANGE     5
#define MAX_SYNTH_PHASE  3600

typedef struct
{
  int16_t freq;    /* four digits used; scale: 0.1; e.g. 1234 -> 123.4 Hz */
  int16_t range;   /* scale factor for freq; 0..MAX_SYNTH_RANGE */
  int16_t phase;   /* -MAX_SYNTH_PHASE..+MAX_SYNTH_PHASE; >0 -> pulses later */
} SYNTH;



/* Time zone/daylight saving parameters. */

/* the name of a time zone, 5 characters plus trailing zero */
typedef char TZ_NAME[6];

typedef struct
{
  int32_t offs;      /* offset from UTC to local time [sec] */
  int32_t offs_dl;   /* additional offset if daylight saving enabled [sec] */
  TM_GPS tm_on;      /* date/time when daylight saving starts */
  TM_GPS tm_off;     /* date/time when daylight saving ends */
  TZ_NAME name[2];   /* names without and with daylight saving enabled */
} TZDL;

/* The constant below is defined beginning with software rev. 1.29. */
/* If the year in tzdl.tm_on and tzdl.tm_off is or'ed with that constant, */
/* the receiver automatically generates daylight saving year by year. */
/* See GPSLIB.TXT for more information. */

#define DL_AUTO_FLAG  0x8000

/* Example: */
/* For automatic daylight saving enable/disable in Central Europe, */
/* the variables are to be set as shown below: */
/*   offs = 3600L           one hour from UTC */
/*   offs_dl = 3600L        one additional hour if daylight saving enabled */
/*   tm_on = first Sunday from March 25, 02:00:00h ( year |= DL_AUTO_FLAG ) */
/*   tm_off = first Sunday from October 25, 03:00:00h ( year |= DL_AUTO_FLAG ) */
/*   name[0] == "CET  "     name if daylight saving not enabled */
/*   name[1] == "CEST "     name if daylight saving is enabled */


// Below there are some initializers for commonly used TZDL configurations:

#define TZ_INFO_UTC  "UTC (Universal Time, Coordinated)"
#define DEFAULT_TZDL_UTC                                                   \
{                                                                          \
  0L,                                                        /* offs */    \
  0L,                                                        /* offs_dl */ \
  { 1994 | DL_AUTO_FLAG, 1, 1, 0, 0, 0, 0, 0, 0L, 0L, 0 },   /* tm_on */   \
  { 1994 | DL_AUTO_FLAG, 1, 1, 0, 0, 0, 0, 0, 0L, 0L, 0 },   /* tm_off */  \
  { "UTC  ", "UTC  " }                                       /* name[] */  \
}

#define TZ_INFO_CET  "CET/CEST (Central Europe)"
#define DEFAULT_TZDL_CET                                                   \
{                                                                          \
  3600L,                                                     /* offs */    \
  3600L,                                                     /* offs_dl */ \
  { 1994 | DL_AUTO_FLAG, 3, 25, 0, 0, 2, 0, 0, 0L, 0L, 0 },  /* tm_on */   \
  { 1990 | DL_AUTO_FLAG, 10, 25, 0, 0, 3, 0, 0, 0L, 0L, 0 }, /* tm_off */  \
  { "CET  ", "CEST " }                                       /* name[] */  \
}


#define TZ_INFO_EET  "EET/EEST (Easter Europe)"
#define DEFAULT_TZDL_EET                                                   \
{                                                                          \
  7200L,                                                     /* offs */    \
  3600L,                                                     /* offs_dl */ \
  { 1994 | DL_AUTO_FLAG, 3, 25, 0, 0, 3, 0, 0, 0L, 0L, 0 },  /* tm_on */   \
  { 1990 | DL_AUTO_FLAG, 10, 25, 0, 0, 4, 0, 0, 0L, 0L, 0 }, /* tm_off */  \
  { "EET  ", "EEST " }                                       /* name[] */  \
}



/*
 * The structure below was defined wit GPS166 v1.31. It reflects the
 * status of the antenna, the times of last disconnect/reconnect,
 * and the board's clock offset after disconnection interval.
 */
typedef struct
{
  int16_t status;      /* current status of antenna */
  TM_GPS tm_disconn;   /* time of antenna disconnect */
  TM_GPS tm_reconn;    /* time of antenna reconnect */
  int32_t delta_t;     /* clock offs. at reconn. time in GPS_TICKS_PER_SEC */
} ANT_INFO;




/* The status field may be set to one of the values below: */

enum
{
  ANT_INVALID,   /* struct not set yet because ant. has not been disconn. */
  ANT_DISCONN,   /* ant. now disconn., tm_reconn and delta_t not set */
  ANT_RECONN     /* ant. has been disconn. and reconn., all fields valid */
};




/* The structure below was defined in rev. 1.47. It holds some flags */
/* which let the corresponding outputs be disabled after power-up until */
/* the receiver has synchronized (flag == 0x00, the default) or force */
/* the outputs to be enabled immediately after power-up. The fixed */
/* frequency output is hard-wired to be enabled immediately after */
/* power-up, so the code for freq must always be 0x03. */

#define EF_OFF            0x00   /* outputs off until sync'd */

#define EF_SERIAL_BOTH    0x03   /* both serial ports on */
#define EF_PULSES_BOTH    0x03   /* both pulses P_SEC and P_MIN on */
#define EF_FREQ_ALL       0x07   /* all fixed freq. outputs on */
#define EF_SYNTH          0x01   /* synth. on */

typedef struct
{
  uint16_t serial;   /* EF_OFF or EF_SERIAL_ON_BOTH */
  uint16_t pulses;   /* EF_OFF or EF_PULSES_ON_BOTH */
  uint16_t freq;     /* always EF_FREQ_ON_ALL */
  uint16_t synth;    /* EF_OFF or EF_SYNTH_ON */
} ENABLE_FLAGS;



/* A struct used to hold the settings of a serial port: */

#ifndef _COM_HS_DEFINED
  /* types of handshake */
  enum { HS_NONE, HS_XONXOFF, HS_RTSCTS };
  #define _COM_HS_DEFINED
#endif

#ifndef _COM_PARM_DEFINED
  typedef int32_t BAUD_RATE;

  /* indices used to identify a parameter in the framing string */
  enum { F_DBITS, F_PRTY, F_STBITS };

  typedef struct
  {
    BAUD_RATE baud_rate;  /* e.g. 19200L */
    char framing[4];      /* e.g. "8N1" */
    int16_t handshake;    /* a numeric value, only HS_NONE supported yet */
  } COM_PARM;

  #define _COM_PARM_DEFINED
#endif



/*
 * Indices of any supported baud rates.
 * Note that not each baud rate must be supported by
 * any clock model and/or port:
 */
enum
{
  MBG_BAUD_RATE_300,
  MBG_BAUD_RATE_600,
  MBG_BAUD_RATE_1200,
  MBG_BAUD_RATE_2400,
  MBG_BAUD_RATE_4800,
  MBG_BAUD_RATE_9600,
  MBG_BAUD_RATE_19200,
  MBG_BAUD_RATE_38400,
  N_MBG_BAUD_RATES       /* the number of supported baud rates */
};

/*
 * An initializer for a table of baud rate values.
 * The values must correspond to the enumeration above.
 */
#define MBG_BAUD_RATES \
{                      \
  300L,                \
  600L,                \
  1200L,               \
  2400L,               \
  4800L,               \
  9600L,               \
  19200L,              \
  38400L               \
}

/*
 * An initializer for a table of baud rate strings.
 * The values must correspond to the enumeration above.
 */
#define MBG_BAUD_STRS \
{                     \
  "300",              \
  "600",              \
  "1200",             \
  "2400",             \
  "4800",             \
  "9600",             \
  "19200",            \
  "38400"             \
}

/*
 * The bit masks below can be used to determine which baud rates
 * are supported by a serial port. This may vary between
 * different ports of the same radio clock since different
 * types of UART are used which must not necessarily support
 * each baud rate:
 */
#define MBG_PORT_HAS_300     ( 1UL << MBG_BAUD_RATE_300 )
#define MBG_PORT_HAS_600     ( 1UL << MBG_BAUD_RATE_600 )
#define MBG_PORT_HAS_1200    ( 1UL << MBG_BAUD_RATE_1200 )
#define MBG_PORT_HAS_2400    ( 1UL << MBG_BAUD_RATE_2400 )
#define MBG_PORT_HAS_4800    ( 1UL << MBG_BAUD_RATE_4800 )
#define MBG_PORT_HAS_9600    ( 1UL << MBG_BAUD_RATE_9600 )
#define MBG_PORT_HAS_19200   ( 1UL << MBG_BAUD_RATE_19200 )
#define MBG_PORT_HAS_38400   ( 1UL << MBG_BAUD_RATE_38400 )


/*
 * Indices of any supported framings.
 * Note that not each framing must be supported by
 * any clock model and/or port:
 */
enum
{
  MBG_FRAMING_7N2,
  MBG_FRAMING_7E1,
  MBG_FRAMING_7E2,
  MBG_FRAMING_8N1,
  MBG_FRAMING_8N2,
  MBG_FRAMING_8E1,
  MBG_FRAMING_7O1,
  MBG_FRAMING_7O2,
  MBG_FRAMING_8O1,
  N_MBG_FRAMINGS       /* the number of supported framings */
};

/*
 * An initializer for a table of framing strings.
 * The values must correspond to the enumeration above.
 */
#define MBG_FRAMING_STRS \
{                        \
  "7N2",                 \
  "7E1",                 \
  "7E2",                 \
  "8N1",                 \
  "8N2",                 \
  "8E1",                 \
  "7O1",                 \
  "7O2",                 \
  "8O1"                  \
}

/*
 * The bit masks below can be used to determine which framings
 * are supported by a serial port. This may vary between
 * different ports of the same radio clock since different
 * types of UART are used which must not necessarily support
 * each framing type:
 */
#define MBG_PORT_HAS_7N2   ( 1UL << MBG_FRAMING_7N2 )
#define MBG_PORT_HAS_7E1   ( 1UL << MBG_FRAMING_7E1 )
#define MBG_PORT_HAS_7E2   ( 1UL << MBG_FRAMING_7E2 )
#define MBG_PORT_HAS_8N1   ( 1UL << MBG_FRAMING_8N1 )
#define MBG_PORT_HAS_8N2   ( 1UL << MBG_FRAMING_8N2 )
#define MBG_PORT_HAS_8E1   ( 1UL << MBG_FRAMING_8E1 )
#define MBG_PORT_HAS_7O1   ( 1UL << MBG_FRAMING_7O1 )
#define MBG_PORT_HAS_7O2   ( 1UL << MBG_FRAMING_7O2 )
#define MBG_PORT_HAS_8O1   ( 1UL << MBG_FRAMING_8O1 )



/*
 * By default, the baud rates and framings below
 * are supported by the UARTs integrated into
 * the C166 microcontroller:
 */
#define DEFAULT_GPS_BAUD_RATES_C166 \
(                                   \
  MBG_PORT_HAS_300   |              \
  MBG_PORT_HAS_600   |              \
  MBG_PORT_HAS_1200  |              \
  MBG_PORT_HAS_2400  |              \
  MBG_PORT_HAS_4800  |              \
  MBG_PORT_HAS_9600  |              \
  MBG_PORT_HAS_19200                \
)

#define DEFAULT_GPS_FRAMINGS_C166   \
(                                   \
  MBG_PORT_HAS_7N2 |                \
  MBG_PORT_HAS_7E1 |                \
  MBG_PORT_HAS_7E2 |                \
  MBG_PORT_HAS_8N1 |                \
  MBG_PORT_HAS_8N2 |                \
  MBG_PORT_HAS_8E1                  \
)


/*
 * By default, the baud rates and framings below
 * are supported by the UARTs integrated into
 * the GP2021 chipset:
 */
#define DEFAULT_GPS_BAUD_RATES_GP2021 \
(                                     \
  MBG_PORT_HAS_300   |                \
  MBG_PORT_HAS_600   |                \
  MBG_PORT_HAS_1200  |                \
  MBG_PORT_HAS_2400  |                \
  MBG_PORT_HAS_4800  |                \
  MBG_PORT_HAS_9600  |                \
  MBG_PORT_HAS_19200                  \
)

#define DEFAULT_GPS_FRAMINGS_GP2021   \
(                                     \
  MBG_PORT_HAS_7N2 |                  \
  MBG_PORT_HAS_7E2 |                  \
  MBG_PORT_HAS_8N1 |                  \
  MBG_PORT_HAS_8E1 |                  \
  MBG_PORT_HAS_8O1                    \
)


/*
 * The structure below is more flexible if different receiver
 * models have different numbers of serial ports, so the old
 * structure PORT_PARM will become obsolete.
 */
typedef struct
{
  COM_PARM parm;        /* speed, framing, etc. */
  uint8_t mode;         /* per second, per minute, etc. */
  uint8_t str_type;     /* type of the output string */
  uint32_t flags;       /* reserved for future use, currently 0 */
} PORT_SETTINGS;


/*
 * The structure below adds an index number to the structure
 * above to allow addressing of several instances:
 */
typedef struct
{
  uint16_t idx;         /* 0..RECEIVER_INFO.n_com_port-1 */
  PORT_SETTINGS port_settings;
} PORT_SETTINGS_IDX;


/*
 * The structure below holds the current settings
 * for a port, plus additional informaton on the
 * port's capabilities. This can be read by setup
 * programs to allow setup of supported features
 * only.
 */
typedef struct
{
  PORT_SETTINGS port_settings;  /* COM port settings as defined above */
  uint32_t supp_baud_rates;  /* bit mask of baud rates supp. by this port */
  uint32_t supp_framings;    /* bit mask of framings supp. by this port */
  uint32_t supp_str_types;   /* bit mask, bit 0 set if str_type[0] supp. */
  uint32_t reserved;         /* reserved for future use, currently 0 */
  uint32_t flags;            /* reserved for future use, currently 0 */
} PORT_INFO;


/*
 * The structure below adds an index number to the structure
 * above to allow addressing of several instances:
 */
typedef struct
{
  uint16_t idx;         /* 0..RECEIVER_INFO.n_com_port-1 */
  PORT_INFO port_info;
} PORT_INFO_IDX;


/*
 * The structure below keeps information for a given
 * string type, e.g. which modes can be used with that
 * string type:
 */
typedef struct
{
  uint32_t supp_modes;  /* bit mask of modes supp. with this string type */
  char long_name[23];   /* long name of the string format */
  char short_name[11];  /* short name of the string format */
  uint16_t flags;       /* reserved, currently always 0 */
} STR_TYPE_INFO;


/*
 * The structure below adds an index number to the structure
 * above to allow addressing of several instances:
 */
typedef struct
{
  uint16_t idx;          /* 0..RECEIVER_INFO.n_str_type-1 */
  STR_TYPE_INFO str_type_info;
} STR_TYPE_INFO_IDX;


/*
 * The codes below define valid modes for time strings,
 * i.e. the condition when a string is being sent
 * via the serial port:
 */
enum
{
  STR_ON_REQ,     /* on request only */
  STR_PER_SEC,    /* automatically if second changes */
  STR_PER_MIN,    /* automatically if minute changes */
  STR_AUTO,       /* automatically if required, e.g. on capture event */
  STR_ON_REQ_SEC, /* if second changes and a request has been received */
  N_STR_MODE      /* the number of valid modes */
};


#define DEFAULT_SHORT_MODE_NAMES \
{                                \
  "'?'",                         \
  "1 sec",                       \
  "1 min",                       \
  "auto",                        \
  "'?' sec"                      \
}


/*
 * Default initializers for English mode string names. Initializers
 * for multi-language strings can be found in pcpslstr.h.
 */
#define ENG_MODE_NAME_STR_ON_REQ       "on request '?' only"
#define ENG_MODE_NAME_STR_PER_SEC      "per second"
#define ENG_MODE_NAME_STR_PER_MIN      "per minute"
#define ENG_MODE_NAME_STR_AUTO         "automatically"
#define ENG_MODE_NAME_STR_ON_REQ_SEC   "sec after request"

#define DEFAULT_ENG_MODE_NAMES   \
{                                \
  ENG_MODE_NAME_STR_ON_REQ,      \
  ENG_MODE_NAME_STR_PER_SEC,     \
  ENG_MODE_NAME_STR_PER_MIN,     \
  ENG_MODE_NAME_STR_AUTO,        \
  ENG_MODE_NAME_STR_ON_REQ_SEC   \
}

/*
 * The definitions below are used to set up bit masks
 * which restrict the modes which can be used with
 * a given string type:
 */
#define MSK_STR_ON_REQ      ( 1UL << STR_ON_REQ )
#define MSK_STR_PER_SEC     ( 1UL << STR_PER_SEC )
#define MSK_STR_PER_MIN     ( 1UL << STR_PER_MIN )
#define MSK_STR_AUTO        ( 1UL << STR_AUTO )
#define MSK_STR_ON_REQ_SEC  ( 1UL << STR_ON_REQ_SEC )


/*
 *  The modes below are supported by most string types:
 */
#define DEFAULT_STR_MODES \
(                         \
  MSK_STR_ON_REQ |        \
  MSK_STR_PER_SEC |       \
  MSK_STR_PER_MIN         \
)


/*
 *  The modes below can be used with the capture string:
 */
#define DEFAULT_STR_MODES_UCAP \
(                              \
  MSK_STR_ON_REQ |             \
  MSK_STR_AUTO                 \
)



/*
 * The number of serial ports which were available
 * with all GPS receiver models:
 */
#define DEFAULT_N_COM   2

/*
 * By default that's also the number of ports
 * currently available:
 */
#ifndef N_COM
  #define N_COM     DEFAULT_N_COM
#endif

/*
 * The structure used to store the modes of both serial ports:
 * (now obsolete)
 */
typedef struct
{
  COM_PARM com[DEFAULT_N_COM];    /* COM0 and COM1 settings */
  uint8_t mode[DEFAULT_N_COM];    /* COM0 and COM1 output mode */
} PORT_PARM;


/*
 * The codes below were used with the obsolete
 * PORT_PARM.mode above. They are defined for
 * compatibility with older devices only:
 */
enum
{
  /* STR_ON_REQ,   defined above */
  /* STR_PER_SEC,  defined above */
  /* STR_PER_MIN,  defined above */
  N_STR_MODE_0 = STR_AUTO,   /* COM0 and COM1 */
  STR_UCAP = N_STR_MODE_0,
  STR_UCAP_REQ,
  N_STR_MODE_1               /* COM1 only */
};



/*
 * The following definitions are used to configure an optional
 * on-board IRIG input or output.
 * A GPS device can generate an IRIG output signal if the bit
 * GPS_HAS_IRIG_TX is set in the RECEIVER_INFO.features field.
 * If a device has an optional IRIG input then the bit 
 * GPS_HAS_IRIG_RX is set.
 */

/*
 * Supported IRIG signal code types:
 *   A002        1000 bps, DC shift, time-of-year
 *   A003        1000 bps, DC shift, time-of-year, SBS
 *   A132        1000 bps, 10 kHz carrier, time-of-year
 *   A133        1000 bps, 10 kHz carrier, time-of-year, SBS
 *   B002        100 bps, DC shift, time-of-year
 *   B003        100 bps, DC shift, time-of-year, SBS
 *   B122        100 bps, 1 kHz carrier, time-of-year
 *   B123        100 bps, 1 kHz carrier, time-of-year, SBS
 *   AFNOR       100 bps, 1 kHz carrier, SBS, complete date
 *   AFNOR DC    100 bps, DC shift, SBS, complete date
 *   IEEE1344    100 bps, 1 kHz carrier, time-of-year, SBS, time zone info
 *   IEEE1344 DC 100 bps, DC shift, time-of-year, SBS, time zone info
 *
 *  time-of-year: day-of-year, hours, minutes, seconds
 *  SBS: straight binary seconds, second-of-day
 *  AFNOR: french standard AFNOR NFS-87500
 */

/*
 * Definitions used with IRIG transmitters which generate
 * the same IRIG code both with and without carrier signal
 * at the same time.
 */
enum
{
  ICODE_TX_B002_B122,
  ICODE_TX_B003_B123,
  ICODE_TX_A002_A132,
  ICODE_TX_A003_A133,
  ICODE_TX_AFNOR,
  ICODE_TX_IEEE1344,
  N_ICODE_TX    /* number of code types */
};


/*
 * Initializers for format name strings.
 */
#define DEFAULT_ICODE_TX_NAMES \
{                              \
  "B002+B122",                 \
  "B003+B123",                 \
  "A002+A132",                 \
  "A003+A133",                 \
  "AFNOR NFS-87500",           \
  "IEEE1344"                   \
}

/*
 * Initializers for English format description strings.
 */
#define DEFAULT_ICODE_TX_DESCRIPTIONS_ENG              \
{                                                      \
  "100 bps, DC or 1 kHz carrier",                      \
  "100 bps, DC or 1 kHz carrier, SBS",                 \
  "1000 bps, DC or 10 kHz carrier",                    \
  "1000 bps, DC or 10 kHz carrier, SBS",               \
  "100 bps, DC or 1 kHz carrier, SBS, complete date",  \
  "100 bps, DC or 1 kHz carrier, SBS, time zone info"  \
}

/*
 * The definitions below are used to set up bit masks
 * which restrict the IRIG formats which are supported
 * by a given IRIG transmitter device:
 */
#define MSK_ICODE_TX_B002_B122    ( 1UL << ICODE_TX_B002_B122 )
#define MSK_ICODE_TX_B003_B123    ( 1UL << ICODE_TX_B003_B123 )
#define MSK_ICODE_TX_A002_A132    ( 1UL << ICODE_TX_A002_A132 )
#define MSK_ICODE_TX_A003_A133    ( 1UL << ICODE_TX_A003_A133 )
#define MSK_ICODE_TX_AFNOR        ( 1UL << ICODE_TX_AFNOR )
#define MSK_ICODE_TX_IEEE1344     ( 1UL << ICODE_TX_IEEE1344 )

/*
 * A mask of IRIG formats with 1 kHz carrier:
 */
#define MSK_ICODE_TX_1KHZ  \
(                          \
  MSK_ICODE_TX_B002_B122 | \
  MSK_ICODE_TX_B003_B123 | \
  MSK_ICODE_TX_AFNOR     | \
  MSK_ICODE_TX_IEEE1344    \
)

/*
 * A mask of IRIG formats with 10 kHz carrier:
 */
#define MSK_ICODE_TX_10KHZ \
(                          \
  MSK_ICODE_TX_A002_A132 | \
  MSK_ICODE_TX_A003_A133   \
)

/*
 * A mask of IRIG formats with 100 bps data rate:
 */
#define MSK_ICODE_TX_100BPS \
(                           \
  MSK_ICODE_TX_B002_B122 |  \
  MSK_ICODE_TX_B003_B123 |  \
  MSK_ICODE_TX_AFNOR     |  \
  MSK_ICODE_TX_IEEE1344     \
)

/*
 * A mask of IRIG formats with 1000 bps data rate:
 */
#define MSK_ICODE_TX_1000BPS \
(                            \
  MSK_ICODE_TX_A002_A132 |   \
  MSK_ICODE_TX_A003_A133     \
)

/*
 * A mask of IRIG formats which support TFOM:
 */
#define MSK_ICODE_TX_HAS_TFOM \
(                             \
  MSK_ICODE_TX_IEEE1344       \
)

/*
 * The default mask of IRIG formats supported by
 * IRIG transmitters:
 */
#if !defined( SUPP_MSK_ICODE_TX )
  #define SUPP_MSK_ICODE_TX  \
  (                          \
    MSK_ICODE_TX_B002_B122 | \
    MSK_ICODE_TX_B003_B123 | \
    MSK_ICODE_TX_A002_A132 | \
    MSK_ICODE_TX_A003_A133 | \
    MSK_ICODE_TX_AFNOR       \
  )
#endif



/*
 * Definitions used with IRIG receivers which decode
 * two similar IRIG codes
 * at the same time.
 */
enum
{
  ICODE_RX_B122_B123,
  ICODE_RX_A132_A133,
  ICODE_RX_B002_B003,
  ICODE_RX_A002_A003,
  ICODE_RX_AFNOR,
  ICODE_RX_AFNOR_DC,
  ICODE_RX_IEEE1344,
  ICODE_RX_IEEE1344_DC,
  N_ICODE_RX          /* the number of valid signal code types */
};

/*
 * Initializers for format name strings.
 */
#define DEFAULT_ICODE_RX_NAMES \
{                              \
  "B122/B123",                 \
  "A132/A133",                 \
  "B002/B003 (DC)",            \
  "A002/A003 (DC)",            \
  "AFNOR NFS-87500",           \
  "AFNOR NFS-87500 (DC)",      \
  "IEEE1344",                  \
  "IEEE1344 (DC)"              \
}

/*
 * Initializers for English format description strings.
 */
#define DEFAULT_ICODE_RX_DESCRIPTIONS_ENG        \
{                                                \
  "100 bps, 1 kHz carrier, SBS optionally",      \
  "1000 bps, 10 kHz carrier, SBS optionally",    \
  "100 bps, DC shift, SBS optionally",           \
  "1000 bps, DC shift, SBS optionally",          \
  "100 bps, 1 kHz carrier, SBS, complete date",  \
  "100 bps, DC shift, SBS, complete date",       \
  "100 bps, 1 kHz carrier, SBS, time zone info", \
  "100 bps, DC shift, SBS, time zone info"       \
}

/*
 * Bit masks corresponding to the enumeration above:
 */
#define MSK_ICODE_RX_B122_B123       ( 1UL << ICODE_RX_B122_B123 )
#define MSK_ICODE_RX_A132_A133       ( 1UL << ICODE_RX_A132_A133 )
#define MSK_ICODE_RX_B002_B003       ( 1UL << ICODE_RX_B002_B003 )
#define MSK_ICODE_RX_A002_A003       ( 1UL << ICODE_RX_A002_A003 )
#define MSK_ICODE_RX_AFNOR           ( 1UL << ICODE_RX_AFNOR )
#define MSK_ICODE_RX_AFNOR_DC        ( 1UL << ICODE_RX_AFNOR_DC )
#define MSK_ICODE_RX_IEEE1344        ( 1UL << ICODE_RX_IEEE1344 )
#define MSK_ICODE_RX_IEEE1344_DC     ( 1UL << ICODE_RX_IEEE1344_DC )

/*
 * A mask of IRIG formats which have DC shift:
 */
#define MSK_ICODE_RX_DC    \
(                          \
  MSK_ICODE_RX_B002_B003 | \
  MSK_ICODE_RX_A002_A003 | \
  MSK_ICODE_RX_AFNOR_DC  | \
  MSK_ICODE_RX_IEEE1344_DC \
)

/*
 * A mask of IRIG formats with 1 kHz carrier:
 */
#define MSK_ICODE_RX_1KHZ  \
(                          \
  MSK_ICODE_RX_B122_B123 | \
  MSK_ICODE_RX_AFNOR     | \
  MSK_ICODE_RX_IEEE1344    \
)

/*
 * A mask of IRIG formats with 10 kHz carrier:
 */
#define MSK_ICODE_RX_10KHZ \
(                          \
  MSK_ICODE_RX_A132_A133   \
)

/*
 * A mask of IRIG formats with 100 bps data rate:
 */
#define MSK_ICODE_RX_100BPS \
(                           \
  MSK_ICODE_RX_B122_B123 |  \
  MSK_ICODE_RX_B002_B003 |  \
  MSK_ICODE_RX_AFNOR     |  \
  MSK_ICODE_RX_AFNOR_DC  |  \
  MSK_ICODE_RX_IEEE1344  |  \
  MSK_ICODE_RX_IEEE1344_DC  \
)

/*
 * A mask of IRIG formats with 1000 bps data rate:
 */
#define MSK_ICODE_RX_1000BPS \
(                            \
  MSK_ICODE_RX_A132_A133 |   \
  MSK_ICODE_RX_A002_A003     \
)

/*
 * A mask of IRIG formats which support TFOM:
 */
#define MSK_ICODE_RX_HAS_TFOM \
(                             \
  MSK_ICODE_RX_IEEE1344 |     \
  MSK_ICODE_RX_IEEE1344_DC    \
)

/*
 * The default mask of IRIG formats supported by
 * IRIG receivers:
 */
#if !defined( SUPP_MSK_ICODE_RX )
  #define SUPP_MSK_ICODE_RX  \
  (                          \
    MSK_ICODE_RX_B122_B123 | \
    MSK_ICODE_RX_A132_A133 | \
    MSK_ICODE_RX_B002_B003 | \
    MSK_ICODE_RX_A002_A003 | \
    MSK_ICODE_RX_AFNOR     | \
    MSK_ICODE_RX_AFNOR_DC    \
  )
#endif


/*
 * The structure below is used to configure an optional
 * on-board IRIG output:
 */
typedef struct
{
  uint16_t icode;   /* IRIG signal code, as enumerated above */
  uint16_t flags;   /* (codes defined below) */
} IRIG_SETTINGS;


/* bit masks used with IRIG_SETTINGS.flags (others are reserved): */

#define IFLAGS_DISABLE_TFOM        0x0001   /* RX ignore/TX don't gen TFOM */
#define IFLAGS_TX_GEN_LOCAL_TIME   0x0002   /* gen local time, not UTC */

#define IFLAGS_MASK                0x0003   /* flags above or'ed */


/*
 * The structure below is used to query the IRIG configuration
 * plus the supported codes:
 */
typedef struct
{
  IRIG_SETTINGS settings;
  uint32_t supp_codes;     /* bit mask of supported codes */
} IRIG_INFO;


typedef struct
{
  uint32_t flags;
} MBG_OPT_SETTINGS;

typedef struct
{
  MBG_OPT_SETTINGS settings;
  uint32_t supp_flags;
} MBG_OPT_INFO;

enum
{
  MBG_OPT_BIT_STR_UTC,   // serial string contains UTC time
  MBG_OPT_BIT_EMU_SYNC,  // emulate/pretend to be synchronized
  N_MBG_OPT_BIT
};

/*
 * Bit masks corresponding to the enumeration above:
 */
#define MBG_OPT_FLAG_STR_UTC   ( 1UL << MBG_OPT_BIT_STR_UTC )
#define MBG_OPT_FLAG_EMU_SYNC  ( 1UL << MBG_OPT_BIT_EMU_SYNC )


/*
 * The structures below are required to setup the programmable
 * pulse outputs which are provided by some GPS receivers.
 * The number of programmable pulse outputs supported by a GPS
 * receiver is reported in the RECEIVER_INFO.n_str_type field.
 */

/*
 * The structure below is used to define a date of year:
 */
typedef struct
{
  uint8_t mday;    /* 1..28,29,30,31 */
  uint8_t month;   /* 1..12 */
  uint16_t year;   /* including century */
} MBG_DATE;


/*
 * The structure below is used to define a time of day:
 */
typedef struct
{
  uint8_t hour;    /* 0..23 */
  uint8_t min;     /* 0..59 */
  uint8_t sec;     /* 0..59,60 */
  uint8_t sec100;  /* reserved, currently always 0 */
} MBG_TIME;


/*
 * The structure below defines a single date and time
 * for switching operations:
 */
typedef struct
{
  MBG_DATE d;    /* date to switch */
  MBG_TIME t;    /* time to switch */
  uint8_t wday;  /* reserved, currently always 0 */
  uint8_t flags; /* reserved, currently 0 */
} MBG_DATE_TIME;


/*
 * The structure below defines times and dates
 * for an on/off cycle:
 */
typedef struct
{
  MBG_DATE_TIME on;   /* time and date to switch on */
  MBG_DATE_TIME off;  /* time and date to switch off */
} POUT_TIME;


/*
 * The number of POUT_TIMEs for each programmable pulse output
 */
#define N_POUT_TIMES 3

/*
 * The structure below is used to configure a single programmable
 * pulse output.
 */
typedef struct
{
  uint16_t mode;        /* mode of operation, codes defined below */
  uint16_t pulse_len;   /* 10 msec units */
  uint16_t timeout;     /* [sec], for dcf_mode */
  uint16_t flags;       /* see below */
  POUT_TIME tm[N_POUT_TIMES];  /* switching times */
} POUT_SETTINGS;


/*
 * The codes below are defined for POUT_SETTINGS.mode to setup
 * the basic mode of operation for a single programmable pulse
 * output:
 */
enum
{
  POUT_IDLE,          /* always off, or on if POUT_INVERTED */
  POUT_TIMER,         /* switch on/off at configured times */
  POUT_SINGLE_SHOT,   /* pulse at time POUT_SETTINGS.tm[0].on */
  POUT_CYCLIC_PULSE,  /* pulse every POUT_SETTINGS.tm[0].on.t interval */
  POUT_PER_SEC,       /* pulse if second changes */
  POUT_PER_MIN,       /* pulse if minute changes */
  POUT_PER_HOUR,      /* pulse if hour changes */
  POUT_DCF77,         /* emulate DCF77 signal */
  POUT_POS_OK,        /* on if pos. OK (nav_solved) */
  POUT_TIME_SYNC,     /* on if time sync (time_syn) */
  POUT_ALL_SYNC,      /* on if pos. OK and time sync */
  N_POUT_MODES
};


/*
 * Default initializers for English pulse mode names. Initializers
 * for multi-language strings can be found in pcpslstr.h.
 */
#define ENG_POUT_NAME_IDLE            "idle"
#define ENG_POUT_NAME_TIMER           "timer"
#define ENG_POUT_NAME_SINGLE_SHOT     "single shot"
#define ENG_POUT_NAME_CYCLIC_PULSE    "cyclic pulse"
#define ENG_POUT_NAME_PER_SEC         "per sec"
#define ENG_POUT_NAME_PER_MIN         "per min"
#define ENG_POUT_NAME_PER_HOUR        "per hour"
#define ENG_POUT_NAME_DCF77           "DCF77 marks"
#define ENG_POUT_NAME_POS_OK          "position OK"
#define ENG_POUT_NAME_TIME_SYNC       "time sync"
#define ENG_POUT_NAME_ALL_SYNC        "all sync"

#define DEFAULT_ENG_POUT_NAMES \
{                              \
  ENG_POUT_NAME_IDLE,          \
  ENG_POUT_NAME_TIMER,         \
  ENG_POUT_NAME_SINGLE_SHOT,   \
  ENG_POUT_NAME_CYCLIC_PULSE,  \
  ENG_POUT_NAME_PER_SEC,       \
  ENG_POUT_NAME_PER_MIN,       \
  ENG_POUT_NAME_PER_HOUR,      \
  ENG_POUT_NAME_DCF77,         \
  ENG_POUT_NAME_POS_OK,        \
  ENG_POUT_NAME_TIME_SYNC,     \
  ENG_POUT_NAME_ALL_SYNC       \
}


/*
 * The definitions below are used to set up bit masks
 * which restrict the modes which can be used with
 * a given programmable output:
 */
#define MSK_POUT_IDLE          ( 1UL << POUT_IDLE )
#define MSK_POUT_TIMER         ( 1UL << POUT_TIMER )
#define MSK_POUT_SINGLE_SHOT   ( 1UL << POUT_SINGLE_SHOT )
#define MSK_POUT_CYCLIC_PULSE  ( 1UL << POUT_CYCLIC_PULSE )
#define MSK_POUT_PER_SEC       ( 1UL << POUT_PER_SEC )
#define MSK_POUT_PER_MIN       ( 1UL << POUT_PER_MIN )
#define MSK_POUT_PER_HOUR      ( 1UL << POUT_PER_HOUR )
#define MSK_POUT_DCF77         ( 1UL << POUT_DCF77 )
#define MSK_POUT_POS_OK        ( 1UL << POUT_POS_OK )
#define MSK_POUT_TIME_SYNC     ( 1UL << POUT_TIME_SYNC )
#define MSK_POUT_ALL_SYNC      ( 1UL << POUT_ALL_SYNC )


/*
 * The codes below are used with POUT_SETTINGS.flags:
 */
#define POUT_INVERTED   0x0001


/*
 * Since a clock may support more than one programmable
 * pulse output, setup tools must use the structure below
 * to read/set pulse output configuration.
 * The number of outputs supported by a receiver model
 * can be queried using the RECEIVER_INFO structure.
 */
typedef struct
{
  uint16_t idx;        /* 0..RECEIVER_INFO.n_prg_out-1 */
  POUT_SETTINGS pout_settings;
} POUT_SETTINGS_IDX;


/*
 * The structure below holds the current settings
 * for a programmable pulse output, plus additional 
 * informaton on the output's capabilities. 
 * This can be read by setup programs to allow setup 
 * of supported features only.
 */
typedef struct
{
  POUT_SETTINGS pout_settings;
  uint32_t supp_modes;   /* bit mask of modes supp. by this output */
  uint32_t reserved;     /* reserved for future use, currently 0 */
  uint32_t flags;        /* reserved for future use, currently 0 */
} POUT_INFO;


/*
 * The structure below adds an index number to the structure
 * above to allow addressing of several instances:
 */
typedef struct
{
  uint16_t idx;          /* 0..RECEIVER_INFO.n_prg_out-1 */
  POUT_INFO pout_info;
} POUT_INFO_IDX;



/* 
 * The codes below are used with devices which support multiple 
 * ref time sources at the same time. The priorities of the
 * supported ref time sources is configurable.
 */


/*
 * All possibly supported ref time sources
 */
enum
{
  MULTI_REF_GPS,            // standard GPS
  MULTI_REF_10MHZ,          // 10 MHz input frequency  
  MULTI_REF_PPS,            // 1 PPS input signal
  MULTI_REF_10MHZ_PPS,      // combined 10 MHz plus PPS
  N_MULTI_REF               // the number of defined sources  
};


/*
 * Names of supported ref time sources
 */
#define DEFAULT_MULTI_REF_NAMES \
{                               \
  "GPS",                        \
  "10 MHz freq in",             \
  "PPS in",                     \
  "10 MHz + PPS in"             \
}


/*
 * Bit masks used to indicate supported reference sources 
 */
#define HAS_MULTI_REF_GPS        ( 1UL << MULTI_REF_GPS )
#define HAS_MULTI_REF_10MHZ      ( 1UL << MULTI_REF_10MHZ )
#define HAS_MULTI_REF_PPS        ( 1UL << MULTI_REF_PPS )
#define HAS_MULTI_REF_10MHZ_PPS  ( 1UL << MULTI_REF_10MHZ_PPS )


/*
 * Up to this number of prioritizable ref input sources are supported
 */
#define N_MULTI_REF_PRIO   4


/*
 * The structure below is used to configure the priority of 
 * the supported ref sources.
 *
 * The number stored in prio[0] of the array indicates the ref time 
 * source with highest priority. If that source fails, the device 
 * falls back to the source indicated by prio[1]. Each field of 
 * the prio[] array must be set to one of the values 0..N_MULTI_REF-1,
 * or to -1 (0xFF) if the value is not assigned.
 */
typedef struct
{
  uint8_t prio[N_MULTI_REF_PRIO];
} MULTI_REF_SETTINGS;



/*
 * The structure below is used to query the MULTI_REF configuration,
 * plus the supported ref sources.
 */
typedef struct
{
  MULTI_REF_SETTINGS settings;  /* current settings */
  uint32_t supp_ref;               /* supp. HAS_MULTI_REF_... codes or'ed */
  uint16_t n_levels;              /* supp. levels, 0..N_MULTI_REF_PRIO */
  uint16_t flags;                 /* reserved, currently 0 */
} MULTI_REF_INFO;



/*------------------------------------------------------------------------*/

/* The structure below is used to communicate with the SCU multiplexer  *
 * board in a redundant system                                          */
 
typedef struct
{
  uint32_t hw_id;                // hardware identification
  uint32_t fw_id;                // firmware identification
  uint32_t flags;                // reserved currently 0
  uint16_t epld_status;          // epld status word, see defintions below
  uint16_t epld_control;         // epld control word, see defintions below
} SCU_STAT_INFO;

typedef struct 
{
  uint16_t epld_control_mask;    // control mask, determines which bit is to be changed
  uint16_t epld_control_value;   // control value, determines value of bits to be changed
  uint32_t flags;                // reserved, currently 0
} SCU_STAT_SETTINGS;

// definitions for status word bit masks
#define MSK_EPLD_STAT_TS1          0x0001   // state of time sync signal clk_1
#define MSK_EPLD_STAT_TS2          0x0002   // state of time sync signal clk_2
#define MSK_EPLD_STAT_TL_ERROR     0x0004   // state of time limit error input
#define MSK_EPLD_STAT_PSU1_OK      0x0008   // state of power supply 1 monitoring input
#define MSK_EPLD_STAT_PSU2_OK      0x0010   // state of power supply 2 monitoring input
#define MSK_EPLD_STAT_AUTO         0x0020   // AUTOMATIC/REMOTE or MANUAL Mode
#define MSK_EPLD_STAT_SEL          0x0040   // select bit for output MUX, ( clk_1 = 0 )
#define MSK_EPLD_STAT_ENA          0x0080   // enable Bit for output MUX, set if enabled
#define MSK_EPLD_STAT_ACO          0x4000   // Access control override bit
#define MSK_EPLD_STAT_WDOG_OK      0x8000   // WDT_OK set to zero if watchdog expired

 
#define MSK_EPLD_CNTL_SEL_REM      0x0800   // remote select for output MUX ( clk_1 = 0 )
#define MSK_EPLD_CNTL_DIS_REM      0x1000   // remote disable for output MUX
#define MSK_EPLD_CNTL_REMOTE       0x2000   // must be set to enable remote operation
#define MSK_EPLD_CNTL_SEL_SNMP     0x4000   // connect COM0 channels to XPORT
#define MSK_EPLD_CNTL_ENA_SNMP     0x8000   // select clk for comm. ( clk1 = 0 )



   



/*------------------------------------------------------------------------*/

/*
 * GPS receiver modes of operation. Some of the codes combinations
 * are obsolete with recent GPS receivers. However, that doesn't
 * matter since the mode is just read from the receiver:
 */
#define REMOTE    0x10
#define BOOT      0x20

#define TRACK     ( 0x01 )
#define AUTO_166  ( 0x02 )
#define WARM_166  ( 0x03          | BOOT )
#define COLD_166  ( 0x04          | BOOT )
#define AUTO_BC   ( 0x05 | REMOTE )
#define WARM_BC   ( 0x06 | REMOTE | BOOT )
#define COLD_BC   ( 0x07 | REMOTE | BOOT )
#define UPDA_166  ( 0x08          | BOOT )
#define UPDA_BC   ( 0x09 | REMOTE | BOOT )

typedef struct
{
  uint16_t mode;
  uint16_t good_svs;
  uint16_t svs_in_view;
  int16_t dac_val;
  int16_t dac_cal;
} STAT_INFO;

#define OSC_DAC_RANGE     4096UL
#define OSC_DAC_BIAS      ( OSC_DAC_RANGE / 2 )
      

#ifndef _IDENT_DEFINED

  typedef union
  {
    char c[16];       // as string which may NOT be terminated
    int16_t wrd[8];
    uint32_t lw[4];
  } IDENT;

  #define _IDENT_DEFINED
#endif


/*
 * The type below is used to configure the length of the 
 * antenna cable:
 */
typedef uint16_t ANT_CABLE_LEN;


/*------------------------------------------------------------------------*/

/* Ephemeris parameters of one specific SV. Needed to compute the position */
/* of a satellite at a given time with high precision. Valid for an */
/* interval of 4 to 6 hours from start of transmission. */

typedef struct
{
  CSUM csum;       /*    checksum of the remaining bytes                  */
  int16_t valid;   /*    flag data are valid                              */

  HEALTH health;   /*    health indication of transmitting SV      [---]  */
  IOD IODC;        /*    Issue Of Data, Clock                             */
  IOD IODE2;       /*    Issue of Data, Ephemeris (Subframe 2)            */
  IOD IODE3;       /*    Issue of Data, Ephemeris (Subframe 3)            */
  T_GPS tt;        /*    time of transmission                             */
  T_GPS t0c;       /*    Reference Time Clock                      [---]  */
  T_GPS t0e;       /*    Reference Time Ephemeris                  [---]  */

  double sqrt_A;   /*    Square Root of semi-major Axis        [sqrt(m)]  */
  double e;        /*    Eccentricity                              [---]  */
  double M0;       /* +- Mean Anomaly at Ref. Time                 [rad]  */
  double omega;    /* +- Argument of Perigee                       [rad]  */
  double OMEGA0;   /* +- Longit. of Asc. Node of orbit plane       [rad]  */
  double OMEGADOT; /* +- Rate of Right Ascension               [rad/sec]  */
  double deltan;   /* +- Mean Motion Diff. from computed value [rad/sec]  */
  double i0;       /* +- Inclination Angle                         [rad]  */
  double idot;     /* +- Rate of Inclination Angle             [rad/sec]  */
  double crc;      /* +- Cosine Corr. Term to Orbit Radius           [m]  */
  double crs;      /* +- Sine Corr. Term to Orbit Radius             [m]  */
  double cuc;      /* +- Cosine Corr. Term to Arg. of Latitude     [rad]  */
  double cus;      /* +- Sine Corr. Term to Arg. of Latitude       [rad]  */
  double cic;      /* +- Cosine Corr. Term to Inclination Angle    [rad]  */
  double cis;      /* +- Sine Corr. Term to Inclination Angle      [rad]  */

  double af0;      /* +- Clock Correction Coefficient 0            [sec]  */
  double af1;      /* +- Clock Correction Coefficient 1        [sec/sec]  */
  double af2;      /* +- Clock Correction Coefficient 2      [sec/sec^2]  */
  double tgd;      /* +- estimated group delay differential        [sec]  */

  uint16_t URA;    /*    predicted User Range Accuracy                    */

  uint8_t L2code;  /*    code on L2 channel                         [---] */
  uint8_t L2flag;  /*    L2 P data flag                             [---] */
} EPH;



/* Almanac parameters of one specific SV. A reduced precision set of */
/* parameters used to check if a satellite is in view at a given time. */
/* Valid for an interval of more than 7 days from start of transmission. */

typedef struct
{
  CSUM csum;       /*    checksum of the remaining bytes                  */
  int16_t valid;   /*    flag data are valid                              */

  HEALTH health;   /*                                               [---] */
  T_GPS t0a;       /*    Reference Time Almanac                     [sec] */

  double sqrt_A;   /*    Square Root of semi-major Axis         [sqrt(m)] */
  double e;        /*    Eccentricity                               [---] */

  double M0;       /* +- Mean Anomaly at Ref. Time                  [rad] */
  double omega;    /* +- Argument of Perigee                        [rad] */
  double OMEGA0;   /* +- Longit. of Asc. Node of orbit plane        [rad] */
  double OMEGADOT; /* +- Rate of Right Ascension                [rad/sec] */
  double deltai;   /* +-                                            [rad] */
  double af0;      /* +- Clock Correction Coefficient 0             [sec] */
  double af1;      /* +- Clock Correction Coefficient 1         [sec/sec] */
} ALM;



/* Summary of configuration and health data of all SVs. */

typedef struct
{
  CSUM csum;               /* checksum of the remaining bytes */
  int16_t valid;           /* flag data are valid */

  T_GPS tot_51;            /* time of transmission, page 51 */
  T_GPS tot_63;            /* time of transmission, page 63 */
  T_GPS t0a;               /* complete reference time almanac */

  CFG cfg[N_SVNO];         /* SV configuration from page 63 */
  HEALTH health[N_SVNO];   /* SV health from pages 51, 63 */
} CFGH;



/* UTC correction parameters */

typedef struct
{
  CSUM csum;       /*    checksum of the remaining bytes                  */
  int16_t valid;   /*    flag data are valid                              */

  T_GPS t0t;       /*    Reference Time UTC Parameters              [sec] */
  double A0;       /* +- Clock Correction Coefficient 0             [sec] */
  double A1;       /* +- Clock Correction Coefficient 1         [sec/sec] */

  uint16_t WNlsf;  /*  week number of nearest leap second                 */
  int16_t DNt;     /*  the day number at the end of which LS is inserted  */
  int8_t delta_tls;  /*                                                   */
  int8_t delta_tlsf; /*                                                   */
} UTC;



/* Ionospheric correction parameters */

typedef struct
{
  CSUM csum;       /*    checksum of the remaining bytes                  */
  int16_t valid;   /*    flag data are valid                              */

  double alpha_0;  /*    Ionosph. Corr. Coeff. Alpha 0              [sec] */
  double alpha_1;  /*    Ionosph. Corr. Coeff. Alpha 1          [sec/deg] */
  double alpha_2;  /*    Ionosph. Corr. Coeff. Alpha 2        [sec/deg^2] */
  double alpha_3;  /*    Ionosph. Corr. Coeff. Alpha 3        [sec/deg^3] */

  double beta_0;   /*    Ionosph. Corr. Coeff. Beta 0               [sec] */
  double beta_1;   /*    Ionosph. Corr. Coeff. Beta 1           [sec/deg] */
  double beta_2;   /*    Ionosph. Corr. Coeff. Beta 2         [sec/deg^2] */
  double beta_3;   /*    Ionosph. Corr. Coeff. Beta 3         [sec/deg^3] */
} IONO;



/* GPS ASCII message */

typedef struct
{
  CSUM csum;       /* checksum of the remaining bytes */
  int16_t valid;   /* flag data are valid */
  char s[23];      /* 22 chars GPS ASCII message plus trailing zero */
} ASCII_MSG;


/* End of header body */


#if defined( _USE_PACK )      // set default alignment
  #pragma pack()
#endif


#endif  /* _GPSDEFS_H */
